# -*- coding: utf-8 -*-

import requests


# TODO: вынести всё что касается проекта в отдельный класс Project, Team...
class CheckmarxSession(object):
    host = ''
    user = ''
    password = ''
    session = None  # requests.Session()
    application_name = 'CheckmarxSession.py'

    def __init__(self, host, user, password, app_name='CheckmarxSession.py'):
        self.host = host + '/CxRestAPI'
        self.user = user
        self.password = password
        self.application_name = app_name
        # init session
        self.session = self._get_session()

    def _get_session(self):

        url = self.host + '/auth/login'
        cred = {
            'userName': self.user,
            'password': self.password,
            # 'grant_type': 'password',
            # 'scope': 'sast_rest_api',
            # 'client_id'
        }

        session = requests.Session()
        session.headers.update({'cxOrigin': self.application_name})
        r = session.post(url, data=cred)
        CheckmarxSession._raise_api_error(r)

        cxCSRFToken = r.cookies.get('CXCSRFToken')
        # cxCookie = r.cookies.get('cxCookie')

        session.headers.update({'CXCSRFToken': cxCSRFToken,
                                'cxOrigin': self.application_name})

        return session

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.session.close()

    @staticmethod
    def _raise_api_error(resp):
        if 400 <= resp.status_code < 500:
            if resp.json() is not None:
                raise requests.HTTPError(u'%s Client Error: %s for url: %s' % (resp.status_code, str(resp.json()), resp.url))
        resp.raise_for_status()

    def get_project_id(self, project_name, team_id, skip_error=False):
        url = self.host + '/projects'
        r = self.session.get(url, params={'projectName': project_name, 'teamId': team_id})
        if not skip_error:
            CheckmarxSession._raise_api_error(r)
        # TODO: Looks like a co-style solution
        if 400 <= r.status_code < 600:
            return None
        return r.json()[0]['id']

    def get_preset_id(self, preset_name):
        url = self.host + '/sast/presets'
        r = self.session.get(url)
        CheckmarxSession._raise_api_error(r)

        for preset in r.json():
            if preset['name'] == preset_name:
                return preset['id']

        raise Exception('Scan preset "{}" not found in checkmarx'.format(preset_name))

    def get_team_id(self, team_name):
        url = self.host + '/auth/teams'
        r = self.session.get(url)
        CheckmarxSession._raise_api_error(r)

        for team in r.json():
            if team['fullName'] == team_name:
                return team['id']

        raise Exception('Team "{}" not found in checkmarx'.format(team_name))

    def get_configuration_id(self, name):
        url = self.host + '/sast/engineConfigurations'
        r = self.session.get(url)
        CheckmarxSession._raise_api_error(r)

        for engineConfig in r.json():
            if engineConfig['name'] == name:
                return engineConfig['id']

        raise Exception('Engine configuration "{}" not found in checkmarx'.format(name))

    def create_project(
        self,
        project_name,
        owning_team_id,
        public=True,
        scan_preset='Checkmarx-ASA',
        engine_configuration='Default Configuration'
    ):
        # Creating project
        url = self.host + '/projects/'

        data = {
            'name': project_name,
            'owningTeam': owning_team_id,
            'isPublic': public,
        }

        r = self.session.post(url, json=data)
        CheckmarxSession._raise_api_error(r)

        project_id = r.json()['id']

        # Configuring project
        if scan_preset is not None and engine_configuration is not None:
            self.configure_project(project_id, scan_preset, engine_configuration)

        return project_id

    def configure_project(self, project_id, scan_preset=None, engine_configuration=None):
        if scan_preset is None and engine_configuration is None:
            return

        cur_preset_id = None
        cur_config_id = None
        if scan_preset is None or engine_configuration is None:
            # TODO: get from current project configuration
            pass

        preset_id = self.get_preset_id(scan_preset) if scan_preset is not None else cur_preset_id
        config_id = self.get_configuration_id(engine_configuration) if engine_configuration is not None else cur_config_id

        url = self.host + '/sast/scanSettings'
        data = {
            'projectId': project_id,
            'presetId': preset_id,
            'engineConfigurationId': config_id,
        }

        r = self.session.post(url, json=data)
        CheckmarxSession._raise_api_error(r)

    def upload(self, project_id, archive):
        from requests_toolbelt.multipart.encoder import MultipartEncoder

        url = self.host + '/projects/{id}/sourceCode/attachments'.format(id=project_id)

        # Stream encoder for reducing memory usage
        data = MultipartEncoder(
            fields={'zippedSource': ('sources.zip', archive, 'application/zip')}
        )

        # Load sources to checkmarx
        r = self.session.post(url, data=data, headers={'Content-Type': data.content_type})
        CheckmarxSession._raise_api_error(r)

    def start_scan(self, project_id, incremental=False, public=True, force=False):
        url = self.host + '/sast/scans'

        data = {
            'projectId': project_id,
            'isIncremental': incremental,
            'isPublic': public,
            'forceScan': force,
        }

        r = self.session.post(url, json=data)
        CheckmarxSession._raise_api_error(r)
        return r.json()['id']
