# -*- coding: utf-8 -*-

import logging
import os
import operator
from zipfile import ZipFile, ZIP_DEFLATED

from sandbox.sdk2 import Task, Vault
from sandbox.sdk2 import parameters, svn
from sandbox.sandboxsdk import environments, process
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
# import sandbox.projects.common.constants as consts


class ScanWithCheckmarxSast(Task):
    """ Task to send project to SAST Checkmarx Accelerator """
    module_logger = logging.getLogger(__name__)

    class Requirements(Task.Requirements):
        # client_tags = ctc.Tag.GENERIC
        environments = (
            environments.PipEnvironment('requests_toolbelt', version='0.8.0'),
            environments.PipEnvironment('formic2', version='1.0.3'),
        )

    class Parameters(Task.Parameters):
        # common parameters
        kill_timeout = 3 * 3600

        # custom parameters
        name = parameters.String(
            'Project Name',
            default='',
            required=True
        )
        arcadia_url = parameters.ArcadiaUrl(
            'Arcadia url'
        )

        with parameters.Group('Checkmarx server') as cx_conn:
            # cx_use_default_settings = parameters.Bool('Use default connection settings') #TODO: implement feature
            cx_host = parameters.String('Checkmarx url', default='http://checkmarx.yandex.net')
            cx_user = parameters.String('Checkmarx user', required=True, description='Checkmarx user to send data to analysis', default='scan_robot')
            vault_cx_pass_owner = parameters.String('Password vault storage owner', required=True, default='CHECKMARX')
            vault_cx_pass_name = parameters.String('Password vault storage name', required=True, default='CHECKMARX_SCAN_ROBOT')

        with parameters.Group('Checkmarx project settings') as cx_settings:
            cx_project = parameters.String('Checkmarx project name', required=True)
            cx_team_name = parameters.String('Checkmarx team of project', required=True, default='\\CxServer\\SP\\Company\\Users')
            cx_exclude_dirs = parameters.String(
                'Comma-seperated list of directories to exclude from scan',
                multiline=True,
                description='Eg. thirdparty libraries, modules, tests, packaging scripts or cvs folders',
                default='_cvs, .bzr, .hg, .idea, .git, .svn, backup, bin, node_modules, obj, tests, test'
            )
            cx_scan_file_patterns = parameters.String(
                'Include/exclude wildcard patterns in ant-style syntax',
                multiline=True,
                description=(
                    'Define the include/exclude wildcard patterns. For example, **/*.java, **/*.html, !**\\test\\**\\XYZ*.\n'
                    'The path will be calculated from arcadia directory. For example, specify "!/*" to exclude ya and ya.bat, but include subdirectories'
                ),
                default='\n'.join([  # arcadia
                    '!/*, !/build/,  !/devtools/, !/fuzzing/, !/junk/, !/contrib/, !/library/, !/util/',
                    '!**/*.DS_Store, !**/*.ipr, !**/*.iws, !**/*.bak, !**/*.tmp,',
                    '!**/*.jar, !**/*.zip, !**/*.rar, !**/*.exe, !**/*.dll, !**/*.pdb, !**/*.7z, !**/*.gz,',
                    '!**/*.tar.gz, !**/*.tar, !**/*.gz, !**/*.ahtm, !**/*.ahtml, !**/*.fhtml, !**/*.hdm,',
                    '!**/*.hdml, !**/*.hsql, !**/*.ht, !**/*.hta, !**/*.htc, !**/*.htd, !**/*.war, !**/*.ear,',
                    '!**/*.htmls, !**/*.ihtml, !**/*.mht, !**/*.mhtm, !**/*.mhtml, !**/*.ssi, !**/*.stm,',
                    '!**/*.stml, !**/*.ttml, !**/*.txn, !**/*.xhtm, !**/*.xhtml, !**/*.class, !**/*.iml, !**/*.dat,',
                    '!**/*.aac, !**/*.aif, !**/*.iff, !**/*.m3u, !**/*.mid, !**/*.mp3,',
                    '!**/*.mpa, !**/*.ra, !**/*.wav, !**/*.wma, !**/*.3g2, !**/*.3gp, !**/*.asf, !**/*.asx,',
                    '!**/*.avi, !**/*.flv, !**/*.mov, !**/*.mp4, !**/*.mpg, !**/*.rm, !**/*.swf, !**/*.vob,',
                    '!**/*.wmv, !**/*.bmp, !**/*.gif, !**/*.jpg, !**/*.png, !**/*.psd, !**/*.tif, !**/*.swf ',
                ])
            )

    def on_execute(self):
        logger = (self.module_logger or logging.getLogger(__name__)).getChild('on_execute')
        logger.info('[>] Entering task %s', self.Parameters.name)

        # parse settings
        cx_host = self.Parameters.cx_host
        cx_user = self.Parameters.cx_user or self.Parameters.vault_cx_pass_owner
        cx_pass = Vault.data(self.Parameters.vault_cx_pass_owner, self.Parameters.vault_cx_pass_name)

        cx_project_name = self.Parameters.cx_project
        cx_team_name = self.Parameters.cx_team_name

        logger.info('----------------------------Configurations:-----------------------------')
        logger.info('Checkmarx host: %s', cx_host)
        logger.info('Checkmarx user: %s', cx_user)

        logger.info('Checkmarx project: %s', cx_project_name)
        logger.info('Checkmarx team: %s', cx_team_name)
        logger.info('Repository to scan: %s', self.Parameters.arcadia_url)
        logger.info('Exclude direcories from scan: %s', self.Parameters.cx_exclude_dirs)
        logger.info('Repository to scan: %s', self.Parameters.cx_scan_file_patterns)
        logger.info('------------------------------------------------------------------------')

        # Защита от дурака, которую надо унести в on_save
        if svn.Arcadia.normalize_url(self.Parameters.arcadia_url) == svn.Arcadia.ARCADIA_TRUNK_URL:
            raise Exception('Please, don\'t scan whole arcadia repo')

        # Getting folder patterns for the list of excluded folders
        includes, excludes = self.parse_patterns(
            file_patterns=str(self.Parameters.cx_scan_file_patterns),
            exclude_dirs=str(self.Parameters.cx_exclude_dirs),
        )

        logger.info('[~] Parsed include file patterns: ' + str(includes))
        logger.info('[~] Parsed exclude file patterns: ' + str(excludes))

        from CheckmarxSession import CheckmarxSession
        # TODO: self.Parameters.name очистить от запрещённых символов.
        APPLICATION_NAME = 'Sandbox.' + ScanWithCheckmarxSast.__name__  # + '.' + self.Parameters.name

        with CheckmarxSession(cx_host, cx_user, cx_pass, app_name=APPLICATION_NAME) as cx:
            team_id = cx.get_team_id(cx_team_name)
            project_id = cx.get_project_id(cx_project_name, team_id)
            logger.info('[*] Creating zip of sources for project id = %s (team id = %s)', project_id, team_id)

            zip_path = self.create_src_zip(self.Parameters.arcadia_url, includes, excludes)

            with open(zip_path, 'rb') as sources:
                cx.upload(project_id, sources)

            scan_id = cx.start_scan(project_id, force=True)
            logger.info('[*] Scan started with id: %s', scan_id)
        logger.info('[<] Task %s finished successfully', self.Parameters.name)

        return 0

    def create_src_zip(self, arcadia_url, includes, excludes):
        logger = (self.module_logger or logging.getLogger(__name__)).getChild('create_src_zip')
        logger.debug('[>] Zipping sources for url %s', arcadia_url)

        # Preparing data to zip:
        src_root = self.obtain_sources(arcadia_url)

        zip_path = 'sources.zip'
        make_zipfile(src_root, zip_path, includes=includes, excludes=excludes)

        logger.debug('[<] Compressed sources to path %s', zip_path)

        return zip_path

    def parse_patterns(self, file_patterns, exclude_dirs):
        logger = (self.module_logger or logging.getLogger(__name__)).getChild('parse_patterns')

        # Getting folder patterns for the list of excluded folders
        folder_patterns = filter(operator.truth, [x.strip().strip('/\\') for x in exclude_dirs.split(',')])
        folder_patterns = set(['**' + os.sep + x + os.sep + '**' for x in folder_patterns])
        logger.debug('[~] Exclude folders patterns: ' + str(folder_patterns))

        # Getting file patterns for the list of file patterns
        excludes = set()
        includes = set()
        for pattern in file_patterns.split(','):
            pattern = pattern.strip()
            if pattern[0] == '!':
                excludes.add(pattern[1:])
            else:
                includes.add(pattern)

        if not includes:
            includes.add('*')

        excludes = folder_patterns.union(excludes)

        return includes, excludes

    def obtain_sources(self, arcadia_url):
        # Get ya tool environment from the same revision and branch
        logger = (self.module_logger or logging.getLogger(__name__)).getChild('obtain_sources')
        logger.debug('[>] Obtaining sources from arcadia')

        project_name = svn.Arcadia.parse_url(arcadia_url).subpath
        arc_root = arcadia_sdk.do_clone(arcadia_url, self)
        process.run_process(
            [
                os.path.join(arc_root, 'ya'),
                'make',
                '--checkout',
                '-j0',
                os.path.join(os.path.relpath(arc_root, os.getcwd()), project_name)
            ],
            log_prefix=logger.name,
            timeout=self.Parameters.kill_timeout,
        )
        # TODO: do_build делает много интересного, в том числе может поменять тип билда.
        # Я прошёлся по коду sdk и вроде это не должно ломать селективный чекаут, но при проблемах надо переключиться на process.run_process.
        # arcadia_sdk.do_build(
        #     source_root=arc_root, targets=[project_name], checkout=True, build_threads=0, build_system=consts.YMAKE_BUILD_SYSTEM, clear_build=False, build_type=consts.DEBUG_BUILD_TYPE
        # )
        arc_version = svn.Arcadia.get_revision(arc_root)
        logger.info('[*] Obtained revision %s for arcadia repo %s', arc_version, arcadia_url)

        logger.debug('[<] Obtained sources from arcadia')

        return arc_root


def make_zipfile(source_dir, zip_path, includes='*', excludes=None):
    from formic import FileSet

    logger = logging.getLogger(__name__).getChild('make_zipfile')

    files = FileSet(include=includes, exclude=excludes, directory=source_dir)
    if not any(files):
        raise Exception('No files found to compress for settings:\nincludes = ')

    relroot = os.path.abspath(source_dir)
    with ZipFile(zip_path, 'w', compression=ZIP_DEFLATED) as zip_file:
        for file in files:
            arcname = os.path.relpath(file, relroot)
            logger.debug('Zipping : ' + arcname)
            zip_file.write(file, arcname)

    zip_size = round(os.path.getsize(zip_path)/float(1024 * 1024), 2)
    logger.info('[~] Zipped files to archive %s, archive size: %s Mb', os.path.abspath(zip_path), zip_size)
