# coding: utf-8

import logging
import time

from sandbox import sdk2
from sandbox.sandboxsdk import environments

from sandbox.common.types import task as ctt
from sandbox.common.types import client as ctc
from sandbox.common.types import resource as ctr
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects.common.build.YaMake2 import YaMake2

logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


class GoldenImageResource(sdk2.Resource):
    any_arch = True
    releasers = ["melkikh"]
    release_subscribers = ["melkikh"]
    releasable = True
    share = True
    restart_policy = ctr.RestartPolicy.DELETE


class YcGoldenImageUpload(sdk2.Task):

    _now = 0

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment("PyJWT", version="1.7.1"),
        ]
        client_tags = ctc.Tag.GENERIC | ctc.Tag.MULTISLOT
        disk_space = 8192
        ram = 8192
        cores = 8

    class Parameters(sdk2.Task.Parameters):

        input_image = sdk2.parameters.Resource(
            label='Resource id to upload to as golden image (otherwise, it will be built from scratch)',
            required=False,
            default_value=None)

        arc_path = sdk2.parameters.String(
            label='Path to image in Arcadia',
            default_value='infra/environments/ycloud-base-bionic-5.4',
            required=False)

        path_to_image = sdk2.parameters.String(
            label='Path to built image',
            default_value='infra/environments/ycloud-base-bionic-5.4/vm-image/rootfs.img',
            required=False)

        upload_image = sdk2.parameters.Bool(
            "Upload image to Yandex.Cloud",
            default=False)

        s3_bucket = sdk2.parameters.String(
            label='Name of S3 bucket',
            default_value='yandexgoldenimage',
            required=False)

        result_image_name = sdk2.parameters.String(
            label='Name of result image',
            default_value='yandex-bionic',
            required=False)

        result_image_description = sdk2.parameters.String(
            label='Description of result image',
            default_value='Ubuntu Bionic (5.4) for Yandex services',
            required=False)

        sa_id = sdk2.parameters.String(
            label='Service Account ID',
            default_value='ajek1e7le6mdsc48eiq1',
            required=False)

        sa_key_id = sdk2.parameters.String(
            label='Service Account key ID',
            default_value='ajeeql37pjfl6jphm2vu',
            required=False)

        s3_key_id = sdk2.parameters.String(
            label='S3 key ID',
            default_value='I4L4FFyoN7cjuoP_x9du',
            required=False)

        folder_id = sdk2.parameters.String(
            label='ID of the folder where will be uploaded image',
            default_value='b1gv004ochjnp8ddk1ea',
            required=False)

        wait_for_operation = sdk2.parameters.Bool(
            "Wait for Compute image creating operataion (otherwirse, task will be finished independently",
            default=True)

        with sdk2.parameters.Output:
            out_image_resource = sdk2.parameters.Resource(
                label='Resource id to upload to Yandex.Cloud',
                required=False,
                default_value=None)

    @property
    def s3client(self):
        from s3 import S3Client
        return S3Client(
            key_id=self.Parameters.s3_key_id,
            access_key=sdk2.Vault.data("GOLDEN_IMAGE_S3_ACCESS_KEY"),
            bucket_name=self.Parameters.s3_bucket
        )

    @property
    def compute(self):
        import jwt  # noqa: F401
        from ycloud import YcImage
        sa_key = {
            'service_account_id': self.Parameters.sa_id,
            'id': self.Parameters.sa_key_id,
            'private_key': sdk2.Vault.data("GOLDEN_IMAGE_SA_PRIVATE_KEY"),
        }
        return YcImage(service_account_key=sa_key)

    @property
    def image_name(self):
        name = self.Parameters.result_image_name
        try:
            revision = str(self.Parameters.out_image_resource.arcadia_revision)
            name += '-r' + revision
        except AttributeError:
            pass

        name += '-sb' + str(self.id)
        # name += '-t' + str(self._now)

        return name

    def on_execute(self):
        self._now = str(int(time.time()))

        if not self.Parameters.input_image:
            with self.memoize_stage.build_image:
                self._build_image()

            build_task = self.find().first()
            logger.debug('Found build task: {}'.format(build_task))
            if build_task.status != ctt.Status.SUCCESS:
                raise SandboxTaskFailureError(
                    'Build in sub-task {} failed'.format(build_task.id))

            image_resource = sdk2.Resource.find(
                GoldenImageResource,
                task=build_task).first()

            logger.debug('Found image resource: {}'.format(image_resource))
            self.Parameters.out_image_resource = image_resource
        else:
            self.Parameters.out_image_resource = self.Parameters.input_image

        if self.Parameters.upload_image:
            self._upload_image()

    def _build_image(self):
        logger.info("Creating YaMake2 child task")
        ya_make = YaMake2(
            self,
            description="Child task for build image (parent: {})".format(
                self.id),
            notifications=self.Parameters.notifications,
            create_sub_task=True,
        )

        ya_make.Requirements.client_tags &= ctc.Tag.LINUX_XENIAL
        ya_make.Parameters.result_rt = GoldenImageResource.name
        ya_make.Parameters.targets = self.Parameters.arc_path
        ya_make.Parameters.arts = self.Parameters.path_to_image
        ya_make.Parameters.result_rd = 'yandex golden-image for YC'
        ya_make.Parameters.result_ttl = '14'
        ya_make.Parameters.use_aapi_fuse = True
        ya_make.Parameters.result_single_file = True
        ya_make.save()

        logger.info("Running YaMake2 child task")
        ya_make.enqueue()
        raise sdk2.WaitTask((ya_make,), ctt.Status.Group.FINISH)

    def _wait_for_operation(self, yc_client, operationId):
        logger.info('Waiting for operation: {}'.format(operationId))
        if not self.Parameters.wait_for_operation:
            return

        yc_client.operation_status(operationId, limit=100)

    def _upload_image(self):
        logger.info("Uploading image to S3")

        image_resource = self.Parameters.out_image_resource
        logger.debug('Found resource to upload: {}'.format(image_resource))

        image_data = sdk2.ResourceData(image_resource)
        image_path = str(image_data.path)
        logger.debug('Got local input resource: {}'.format(image_path))

        logger.info("Uploading image to S3")
        try:
            self.s3client.upload(image_path, self.image_name)
            image_url = self.s3client.create_presigned_url(self.image_name)
        except Exception as e:
            logger.exception("Failed to upload image {} to {}".format(
                self.image_name, image_path))
            raise e
        logger.debug('Image was uploaded with temp url: {}'.format(image_url))

        if not image_url:
            raise ValueError('image URL should not be empty')

        logger.info('Uploading image to YC.Compute')
        with self.compute as yc:
            resp = yc.create(
                uri=image_url,
                folderId=self.Parameters.folder_id,
                name=self.image_name,
                description=self.Parameters.result_image_description,
            ).json()

            logger.debug(resp)

            operataionId = resp.get('id')
            if operataionId == '':
                raise ValueError('operataionId should not be empty')

            self._wait_for_operation(yc, operataionId)
            logger.info('Image was successfully uploaded to YC.Compute')
