# coding: utf-8

import logging
import requests
import time
from datetime import datetime
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry


logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__name__)


def requests_retry_session(
    retries=3,
    backoff_factor=0.3,
    session=None,
):
    session = session or requests.Session()
    retry = Retry(
        total=retries,
        read=retries,
        connect=retries,
        backoff_factor=backoff_factor,
        method_whitelist=False  # retry on any method
    )
    # hack
    retry.RETRY_AFTER_STATUS_CODES = frozenset([413, 429, 503, 502])
    adapter = HTTPAdapter(max_retries=retry)
    session.mount("http://", adapter)
    session.mount("https://", adapter)

    return session


class YcImage(object):

    __compute_api = 'compute.api.cloud.yandex.net'
    __iam_api = 'iam.api.cloud.yandex.net'
    __operation_api = 'operation.api.cloud.yandex.net'
    __SECONDS_IN_HOUR = 60. * 60.
    __1GB_IN_BYTES = 1024 ** 3

    def __init__(self, service_account_key):
        self.session = None
        self.__sa_key = service_account_key

    def __enter__(self):
        self.session = requests_retry_session()
        self.session.headers.update(
            {"Authorization": "Bearer {}".format(self.__auth())})
        return self

    def __exit__(self, exception_type, exception_value, traceback):
        pass

    def __prepare_jwt(self):
        import jwt
        now = time.time()
        now_utc = datetime.utcfromtimestamp(now)
        exp_utc = datetime.utcfromtimestamp(now + self.__SECONDS_IN_HOUR)
        payload = {
            "iss": self.__sa_key["service_account_id"],
            "aud": "https://iam.api.cloud.yandex.net/iam/v1/tokens",
            "iat": now_utc,
            "exp": exp_utc,
        }

        headers = {
            "typ": "JWT",
            "alg": "PS256",
            "kid": self.__sa_key["id"],
        }

        return jwt.encode(payload, self.__sa_key["private_key"], algorithm="PS256", headers=headers)

    def __auth(self):
        logger.info("trying to exchange JWT to IAM-token")
        resp = self.session.post('https://{}/{}'.format(
            self.__iam_api, 'iam/v1/tokens'), json={'jwt': self.__prepare_jwt()})
        resp.raise_for_status()
        return resp.json().get('iamToken', '')

    def list(self, folderId):
        logger.info("trying to list images in folder: %s", folderId)
        resp = self.session.get('https://{}/{}'.format(
            self.__compute_api, 'compute/v1/images'), params={'folderId': folderId})
        resp.raise_for_status()
        return resp.json()

    def create(self, uri, folderId, name, description, family="yandex-golden-image", minDiskSize=4*__1GB_IN_BYTES):
        image = {
            "folderId": folderId,
            "name": name,
            "description": description,
            "labels": {},
            "family": family,
            "minDiskSize": str(minDiskSize),
            "os": {
                "type": "LINUX"
            },
            "uri": str(uri),
        }

        logger.info("trying to create image: %r", image)
        resp = self.session.post(
            'https://{}/{}'.format(self.__compute_api, 'compute/v1/images'), json=image)
        resp.raise_for_status()
        return resp

    def operation_status(self, operationId, limit=100):
        counter = 0
        while True:
            logger.info("fetching operation (%s) status", operationId)
            resp = self.session.get('https://{}/{}/{}'.format(
                self.__operation_api, 'operations', str(operationId))).json()
            if resp.get('done', False):
                return
            elif counter < limit:
                counter += 1
                time.sleep(5)
            else:
                raise Exception(
                    'Timed out for operation {}'.format(operationId))
