import itertools
import multiprocessing
from yt.wrapper import YtClient
import yt.wrapper as yt
from sandbox import sdk2
import requests
import json
from time import sleep
# import urllib


class DeployParser():
    def __init__(self, yt_oauth, yp_oauth, idm_oauth):
        self.yt_oauth = yt_oauth
        self.yp_oauth = yp_oauth
        self.idm_oauth = idm_oauth

        self.requests = requests.Session()
        self.requests.headers.update({'Authorization': f'OAuth {yp_oauth}'})
        self.requests.headers.update({'Content-Type': 'application/json'})
        self.requests.headers.update({'Accept': 'application/json, text/plain, */*'})

        self.yt_client = YtClient(proxy="hahn", token=yt_oauth)

    def get_projects(self) -> list:
        body = '{"object_type":"project","selector":{"paths":["/meta/id","/labels/tags"]},"format":1,"options":{"limit":999999,"fetch_timestamps":false}}'
        r = self.requests.post('https://xdc.yp.yandex-team.ru:8443/ObjectService/SelectObjects', json=json.loads(body))
        json_response = r.json()
        results = json_response['results']
        return [i['value_payloads'][0]['yson'] for i in results]

    def get_stages_by_project(self, project):
        body = """
        {"object_type":"stage","selector":{"paths":["/meta/id","/status"]},"filter":{"query":"[/labels/deploy_engine]='env_controller' AND [/meta/project_id] = '""" + project + """'"},"format":1,"options":{"limit":50,"fetch_timestamps":true}}
        """
        r = self.requests.post('https://xdc.yp.yandex-team.ru:8443/ObjectService/SelectObjects', json=json.loads(body))
        json_response = r.json()
        stages = ()
        if 'results' in json_response:
            results = json_response['results']
            stages = tuple(i['value_payloads'][0]['yson'] for i in results)
        return stages

    def is_sox_service(self, stage):
        body = '{"object_type":"stage","selector":{"paths":["/meta","/labels","/spec","/status"]},"subrequests":[{"object_id":"' + stage + '"}],"format":1,"options":{"fetch_timestamps":false}}'
        r = self.requests.post('https://xdc.yp.yandex-team.ru:8443/ObjectService/GetObjects', json=json.loads(body))
        # json_response = r.json()
        is_sox = True if '"sox_service":true' in r.text else False
        return {'stage': stage, 'sox': is_sox}

    def list_merge(self, lstlst):
        all = []
        for lst in lstlst:
            all.extend(lst)
        return all

    def get_tags_by_stage(self, stage):
        body = '{"object_type":"stage","selector":{"paths":["/labels"]},"subrequests":[{"object_id":"' + stage + '"}],"format":1,"options":{"fetch_timestamps":false}}'
        r = self.requests.post('https://xdc.yp.yandex-team.ru:8443/ObjectService/GetObjects', json=json.loads(body))
        if json_response := r.json():
            tags = []
            if 'tags' in json_response['subresponses'][0]['result']['value_payloads'][0]['yson']:
                tags = json_response['subresponses'][0]['result']['value_payloads'][0]['yson']['tags']
            return tags

    def get_maintainers(self, project):
        headers = {'Authorization': f'OAuth {self.idm_oauth}'}
        json_resp = self.requests.get(
            f'https://idm-api.yandex-team.ru/api/v1/roles/?offset=0&limit=20&order_by=-updated&system=deploy-prod&path=%2F{project}%2FMAINTAINER%2F&no_meta=true', headers=headers).json()
        objects = json_resp['objects']

        result = []
        for object in objects:
            if object['group']:
                result.append(f'group:{object["group"]["id"]}')
            if object['user']:
                result.append(object["user"]["username"])
        return result

    def get_owners(self, project):
        headers = {'Authorization': f'OAuth {self.idm_oauth}'}
        json_resp = self.requests.get(
            f'https://idm-api.yandex-team.ru/api/v1/roles/?offset=0&limit=20&order_by=-updated&system=deploy-prod&path=%2F{project}%2FOWNER%2F&no_meta=true', headers=headers).json()
        objects = json_resp['objects']

        result = []
        for object in objects:
            if object['group']:
                result.append(f'group:{object["group"]["id"]}')
            if object['user']:
                result.append(object["user"]["username"])
        return result

    def get_acl_by_project(self, project):
        maintainers, owners = self.get_maintainers(project), self.get_owners(project)
        return {'owner': owners, 'maintainer': maintainers}

    """
    def get_acl_by_project(self, project):
        json_resp = self.requests.get(f'https://deploy-auth.yandex-team.ru/api/get-project-acl?project_id={project}').json()
        return json_resp['project_acl'][project]
    """

    """
    def get_group_members_by_abc(self, group_id):
        body = urllib.parse.urlencode({
            'service': str(group_id),
            'unique': True,
            'is_robot': False,
            'fields': 'person.uid,person.login'
        })
        r = self.requests.get(f'https://abc-back.yandex-team.ru/api/v4/services/members/?{body}')
        import logging
        logging.info(r.url)
        json_resp = r.json()['results']
        logins = [i['person']['login'] for i in json_resp]
        uids = [i['person']['uid'] for i in json_resp]
        result = list(zip(uids, logins))
        return result
    """

    def get_group_members_by_abc(self, group_id):
        try:
            json_resp = self.requests.get(
                f'https://staff-api.yandex-team.ru/v3/persons/?_fields=login,uid&groups.group.id={group_id}&_limit=9999').json()
            members = json_resp['result']
            import logging
            logging.info(f'https://staff-api.yandex-team.ru/v3/persons/?_fields=login,uid&groups.group.id={group_id}&_limit=9999')

            uids = [member['uid'] for member in members]
            logins = [member['login'] for member in members]
            return list(zip(uids, logins))
        except:
            sleep(1)
            self.get_group_members_by_abc(group_id)

    def get_uid_by_login(self, login):
        json_resp = self.requests.get(f'https://staff-api.yandex-team.ru/v3/persons/?_fields=login,uid&login={login}').json()
        member = json_resp['result'][0]
        return (member['uid'], member['login'])

    def get_acl_users_by_project(self, project):
        acl = self.get_acl_by_project(project)

        owners = list(map(lambda owner: owner.replace('group:', ''), acl['owner']))
        maintainers = list(map(lambda owner: owner.replace('group:', ''), acl['maintainer']))

        all_acl = list(itertools.chain(owners, maintainers))
        acl_abc = list(filter(lambda owner: True if owner.isnumeric() else False, all_acl))
        acl_users = list(filter(lambda owner: False if owner.isnumeric() else True, all_acl))

        all_users = list(map(self.get_group_members_by_abc, acl_abc))
        all_users = [item for sublist in all_users for item in sublist] + list(map(self.get_uid_by_login, acl_users))
        all_users = [t for t in (set(tuple(i) for i in all_users))]
        return all_users

    def get_acl_by_stage(self, stage):
        body = '{"object_type":"stage","selector":{"paths":["/meta","/labels","/spec","/status"]},"subrequests":[{"object_id":"' + stage + '"}],"format":1,"options":{"fetch_timestamps":false}}'
        json_response = self.requests.post('https://xdc.yp.yandex-team.ru:8443/ObjectService/GetObjects', json=json.loads(body)).json()
        project = json_response['subresponses'][0]['result']['value_payloads'][0]['yson']['project_id']
        return self.get_acl_users_by_project(project)

    def save_to_yt(self, stage_acl_data, path):
        schema = [
            {"name": "abc", "type": "string"},
            {"name": "nanny_id", "type": "string"},
            {"name": "pod_set_id", "type": "string"},
            {"name": "staff", "type": "string"},
            {"name": "stage", "type": "string"},
            {"name": "user", "type": "string"},
            {"name": "user_id", "type": "int64"}
        ]
        result = []
        for stage, acl in stage_acl_data:
            for user_id, user_login in acl:
                result.append({'abc': None, 'nanny_id': None, 'pod_set_id': None, 'staff': None, 'stage': stage,
                               'user': user_login, 'user_id': int(user_id)})
        self.yt_client.write_table(
            yt.TablePath(path, schema=schema),
            result,
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        )

    def deploy_parse(self, path):
        pool = multiprocessing.Pool(25)

        projects = self.get_projects()
        stages = self.list_merge(list(pool.map(self.get_stages_by_project, projects)))
        tags = list(pool.map(self.get_tags_by_stage, stages))

        prod_stages_with_tags = list(filter(lambda item: 'prod' in item[1], [i for i in list(zip(stages, tags))]))
        prod_stages_with_names = list(filter(lambda item: 'prod' in item[0], [i for i in list(zip(stages, tags))]))

        prod_stages = list(set([i[0] for i in prod_stages_with_tags] + [i[0] for i in prod_stages_with_names]))

        acl = list(pool.map(self.get_acl_by_stage, prod_stages))
        result = list(zip(prod_stages, acl))
        self.save_to_yt(result, path)


class DEPLOY_ACL_PARSER_TASK(sdk2.Task):

    class Parameters(sdk2.Parameters):
        enable_yav = True
        app_tokens = sdk2.parameters.YavSecret("APP OAuth", default="sec-01fynr8vy3w0cbsx7ag0gbhea0")
        acl_path = sdk2.parameters.String(
            "Путь к таблице на YT, в которую будет сохранен ACL на основе парсинга Deploy",
            default="//home/infrasec/gideon-analytics/resources/acl-deploy"
        )

    def on_execute(self):
        app_tokens = self.Parameters.app_tokens.data()

        yt_oauth = app_tokens["yt_oauth"]
        yp_oauth = app_tokens["app_oauth"]
        idm_oauth = app_tokens["idm_oauth"]

        acl_path = self.Parameters.acl_path

        deploy_parser = DeployParser(yt_oauth=yt_oauth, yp_oauth=yp_oauth, idm_oauth=idm_oauth)
        deploy_parser.deploy_parse(acl_path)
