import requests
import urllib
import multiprocessing
from yt.wrapper import YtClient
from sandbox import sdk2
import yt.wrapper as yt


def list_merge(list_of_lists):
    result = []
    for list in list_of_lists:
        result.extend(list)
    return result


class NannyParser():
    def __init__(self, ga_oauth, yt_oauth, nanny_oauth, acl_path):
        self.ga_oauth = ga_oauth
        self.yt_oauth = yt_oauth
        self.acl_path = acl_path
        self.nanny_oauth = nanny_oauth

        self.requests = requests.Session()
        self.requests.headers = {'Authorization': 'OAuth ' + self.ga_oauth}
        self.requests.headers.update({'Content-Type': 'application/json'})
        self.requests.headers.update({'Accept': 'application/json, text/plain, */*'})

        self.yt_client = YtClient(proxy="hahn", token=yt_oauth)

    def get_members_by_abc(self, id):
        body = urllib.parse.urlencode({
            'service': str(id),
            'unique': True,
            'is_robot': False,
            'fields': 'person.uid,person.login'
        })
        r = self.requests.get(f'https://abc-back.yandex-team.ru/api/v4/services/members/?{body}')
        json_resp = r.json()['results']
        logins = [i['person']['login'] for i in json_resp]
        uids = [i['person']['uid'] for i in json_resp]
        result = list(zip(logins, uids))
        return result

    def get_members_by_abc_staff(self, id):
        body = urllib.parse.urlencode({
            '_fields': 'login,name,uid',
            'groups.group.id': str(id)
        })
        r = self.requests.get(f'https://staff-api.yandex-team.ru/v3/persons/?{body}')
        json_resp = r.json()['result']
        logins = [i['login'] for i in json_resp]
        uids = [i['uid'] for i in json_resp]
        result = list(zip(logins, uids))
        return result

    def get_uid_by_member(self, login):
        body = urllib.parse.urlencode({
            'person__login': login,
            'fields': 'person.uid',
            'page_size': 1
        })
        r = self.requests.get(f'https://abc-back.yandex-team.ru/api/v4/services/members/?{body}')
        if not r.json()['results']:
            return (login, 0)
        uid = r.json()['results'][0]['person']['uid']
        return (login, uid)

    def get_nanny_services_from_yp(self):
        body = {
            "object_type": "group",
            "selector": {
                "paths": [
                    "/labels/service",
                    "/spec/members",
                ]
            },
            "filter": {
                "query": "[/labels/system]='nanny' AND (is_substr('prod', string([/labels/service])) OR is_substr("
                         "'stable', string([/labels/service]))) AND NOT (is_substr('preprod', "
                         "string([/labels/service])) OR is_substr('stable', string([/labels/service]))) "
            },
            "format": 1,
            "options": {
                "limit": 99999,
                "offset": 1,
                "fetch_timestamps": False
            }
        }
        result = {}
        for dc in ['iva', 'myt', 'sas', 'man', 'vla']:
            r = self.requests.post(f'https://{dc}.yp.yandex-team.ru:8443/ObjectService/SelectObjects', json=body)
            json_resp = r.json()['results']
            for i in json_resp:
                service_name = i['value_payloads'][0]['yson']
                acl = i['value_payloads'][1]['yson']
                if acl is not None:
                    if service_name not in result:
                        result[service_name] = [acl]
                    else:
                        result[service_name].append(acl)
        to_return_format = []
        for service in result:
            result[service] = list_merge(result[service])
            to_return_format.append((service, result[service]))
        return to_return_format

    def normalize_acl(self, services):
        result_acl = []
        service_name, acl_list = services[0], services[1]
        for acl_value in list(set(acl_list)):
            acl = []
            flag = False
            if acl_value.startswith('staff:') or acl_value.startswith('robot'):
                continue
            if acl_value.startswith('abc:'):
                if 'abc:service-scope:' in acl_value:
                    acl_value = acl_value.split(':')[-2]
                for i in self.get_members_by_abc(acl_value.lstrip('abc:service:')):
                    acl.append(i)
                flag = True
            if not flag:
                acl.append(self.get_uid_by_member(acl_value))
            result_acl.append(acl)
        return (service_name, list_merge(result_acl))

    def parse_services_from_yp(self):
        services = self.get_nanny_services_from_yp()
        try:
            pool = multiprocessing.Pool(25)
            services = pool.map(self.normalize_acl, services)
        finally:
            pool.close()
            pool.join()
        return services

    def save_to_yt(self, services, path):
        schema = [
            {"name": "abc", "type": "string"},
            {"name": "nanny_id", "type": "string"},
            {"name": "pod_set_id", "type": "string"},
            {"name": "staff", "type": "string"},
            {"name": "stage", "type": "string"},
            {"name": "user", "type": "string"},
            {"name": "user_id", "type": "int64"}
        ]
        to_upload = []
        for service, acl_list in services:
            for acl_item in acl_list:
                login, uid = acl_item[0], int(acl_item[1])
                to_upload_value = {'abc': None, 'nanny_id': service, 'pod_set_id': None, 'staff': None, 'stage': None,
                                   'user': login, 'user_id': uid}
                to_upload.append(to_upload_value)
        self.yt_client.write_table(
            yt.TablePath(path, schema=schema),
            to_upload,
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        )

    def get_nanny_services_from_api(self):
        json_payload = {
            "includeLabels": False,
            "storageMode": "NONE"
        }
        r = self.requests.post('https://nanny.yandex-team.ru/api/repo/ListSummaries/', json=json_payload)
        json_response = r.json()['value']
        services_list = [i['serviceId'] for i in json_response]
        return services_list

    def get_service_roles_from_nanny(self, service):
        r = self.requests.get(f'https://nanny.yandex-team.ru/v2/services/{service}/', headers={'Authorization': 'OAuth ' + self.nanny_oauth})
        json_resp = r.json()
        auth_attrs = json_resp['auth_attrs']['content']
        # avaible_roles = [item for item in auth_attrs if item != 'observers']
        avaible_roles = ['owners', 'conf_managers', 'ops_managers']
        def_logins = list_merge([auth_attrs[item]['logins'] for item in avaible_roles])
        groups = list_merge([auth_attrs[item]['groups'] for item in avaible_roles])
        return (service, def_logins + groups)

    def parse_services_from_api(self, services):
        try:
            pool = multiprocessing.Pool(25)
            services_with_acl = pool.map(self.get_service_roles_from_nanny, services)
        finally:
            pool.close()
            pool.join()
        result = []
        for service, acl in services_with_acl:
            abc_groups = list(filter(lambda acl_value: acl_value.isnumeric(), acl))
            clear_members = list(filter(lambda acl_value: not acl_value.isnumeric(), acl))
            try:
                pool = multiprocessing.Pool(25)
                group_members = list_merge(pool.map(self.get_members_by_abc_staff, abc_groups))
            finally:
                pool.close()
                pool.join()
            try:
                pool = multiprocessing.Pool(25)
                members = pool.map(self.get_uid_by_member, clear_members)
            finally:
                pool.close()
                pool.join()
            result.append((service, group_members + members))
        return result


class NANNY_ACL_PARSER_TASK(sdk2.Task):

    class Parameters(sdk2.Parameters):
        enable_yav = True
        app_tokens = sdk2.parameters.YavSecret("APP OAuth", default="sec-01fynr8vy3w0cbsx7ag0gbhea0")
        acl_path = sdk2.parameters.String(
            "Путь к таблице на YT, в которую будет сохранен ACL на основе парсинга Nanny",
            default="//home/infrasec/gideon-analytics/resources/acl-nanny"
        )

    def on_execute(self):
        ga_oauth = self.Parameters.app_tokens.data()["app_oauth"]
        nanny_oauth = self.Parameters.app_tokens.data()['nanny_oauth']
        yt_oauth = self.Parameters.app_tokens.data()['yt_oauth']
        acl_path = self.Parameters.acl_path
        nanny_client = NannyParser(ga_oauth, yt_oauth, nanny_oauth, acl_path)
        nanny_services_from_yt = nanny_client.parse_services_from_yp()
        raw_nanny_services_from_yt = [i[0] for i in nanny_services_from_yt]
        nanny_services_from_api = nanny_client.get_nanny_services_from_api()
        gencfg_services = list(set(nanny_services_from_api) - set(raw_nanny_services_from_yt))
        prod_gencfg_services = list(filter(lambda service: ('prod' in service or 'stable' in service) and not ('preprod' in service or 'prestable' in service), gencfg_services))
        nanny_services_from_api = nanny_client.parse_services_from_api(prod_gencfg_services)
        all_services = nanny_services_from_yt + nanny_services_from_api
        nanny_client.save_to_yt(all_services, acl_path)
