from sandbox import sdk2
import multiprocessing as multproc
from yt.wrapper import YtClient
import yt.wrapper as yt
import requests


class StaffParser():
    def __init__(self, staff_oauth, yt_oauth):
        self.staff_oauth = staff_oauth
        self.yt_oauth = yt_oauth

        self.requests = requests.Session()
        self.requests.headers['Authorization'] = f'OAuth {staff_oauth}'

        self.yt_client = YtClient(proxy="hahn", token=yt_oauth)

    def requests_json(self, url, body=None):
        response = self.requests.post(url, body) if body else requests.get(url)
        return response.json()

    def get_pod_by_login(self, login):
        url = 'https://staff-api.yandex-team.ru/v3/groupmembership'
        body = {
            '_pretty': '1',
            '_fields': 'group',
            'person.login': login,
            'group.type': 'department'
        }
        response_json = self.requests_json(url, body)
        result = response_json['result'][0]
        ancestors = result['group']['ancestors']
        pod = ' → '.join([i['name'] for i in ancestors])
        if not pod:
            pod = result['group']['name']
        if 'name' in result['group'] and result['group']['name'] != pod:
            pod += f" → {result['group']['name']}"
        return pod

    def get_full_departmentstaff(self, limit):
        url = 'https://staff-api.yandex-team.ru/v3/persons'
        body = {
            'official.is_robot': 'false',
            'is_deleted': 'false',
            'department_group.type': 'department',
            'department_group.is_deleted': 'false',
            'official.is_dismissed': 'false',
            'official.affiliation': 'yandex',
            '_limit': limit
        }
        json_response = self.requests_json(url, body)
        result = json_response['result']
        persons_info = {}
        for person in result:
            first_name = person['name']['first']['ru']
            last_name = person['name']['last']['ru']
            full_name = ' '.join([first_name, last_name])
            login = person['login']
            if login is not None:
                persons_info.setdefault(
                    login, {
                        'full_name': full_name
                    }
                )
        return persons_info

    def parse_pods_from_stuff(self, departmentstaff):
        logins = [i for i in departmentstaff]
        with multproc.Pool(20) as pool:
            result = pool.map(self.get_pod_by_login, logins)
        for i, login in enumerate(departmentstaff):
            departmentstaff[login].setdefault('pod', result[i])
        return departmentstaff

    def get_total_from_departmentstaff(self):
        url = 'https://staff-api.yandex-team.ru/v3/persons'
        body = {
            'official.is_robot': 'false',
            'is_deleted': 'false',
            'department_group.type': 'department',
            'department_group.is_deleted': 'false',
            'official.is_dismissed': 'false',
            'official.affiliation': 'yandex',
            '_limit': 1
        }
        json_response = self.requests_json(url, body)
        total = json_response['total']
        return total

    def save_departmentstaff_to_yt(self, departmentstaff, path):
        schema = [
            {"name": "full_name", "type": "string"},
            {"name": "login", "type": "string"},
            {"name": "pod_full", "type": "string"},
            {"name": "pod_short", "type": "string"}
        ]
        result = []
        for login in departmentstaff:
            result.append({
                'full_name': departmentstaff[login]['full_name'],
                'pod_full': departmentstaff[login]['pod'],
                'pod_short': departmentstaff[login]['pod'].split('→')[-1].strip(),
                'login': login,
            })
        self.yt_client.write_table(
            yt.TablePath(path, schema=schema),
            result,
            format=yt.JsonFormat(attributes={"encode_utf8": False})
        )


class STUFF_PARSER_TASK(sdk2.Task):

    class Parameters(sdk2.Parameters):
        enable_yav = True
        app_tokens = sdk2.parameters.YavSecret("APP OAuth", default="sec-01fynr8vy3w0cbsx7ag0gbhea0")
        yt_result_path = sdk2.parameters.String("Путь к таблице на YT, в которую будет сохранен дамп стаффа", default="//home/infrasec/gideon-analytics/resources/staff-dump")

    def on_execute(self):
        app_tokens = self.Parameters.app_tokens.data()

        staff_oauth = app_tokens["app_oauth"]
        yt_oauth = app_tokens["yt_oauth"]

        yt_result_path = self.Parameters.yt_result_path

        staff_client = StaffParser(staff_oauth, yt_oauth)
        total_members_count = staff_client.get_total_from_departmentstaff()
        departmentstaff_without_pods = staff_client.get_full_departmentstaff(total_members_count)
        departmentstaff_with_pods = staff_client.parse_pods_from_stuff(departmentstaff_without_pods)
        staff_client.save_departmentstaff_to_yt(departmentstaff_with_pods, yt_result_path)
