#!/usr/bin/env python

import logging
import os
import shlex
import subprocess as sp


TEST_MAKE_QUARANTINE_PATH = "/home/procenkoeg/arcadia/logfeller/python/logfeller/bin/standalone/make_quarantine/make-quarantine"
TEST_LOG_FELLER_SIMPLE_YT_PARSER = "/home/procenkoeg/arcadia/logfeller/python/logfeller/bin/standalone/make_quarantine/logfeller-simple-yt-parser"

YT_TOKEN = os.getenv("YT_TOKEN")


class QuarantineMaker(object):
    """
    Get yt stream, parse, serch secrets and create some tables.

    Example:
        ```
        yt_account = "security"
        stream_name = "xiva-xivaserver-access-log"
        event_time = "2020-06-13T03:10:00Z"
        create_quarantine_table_command = " ".join([
            "./make-quarantine", "--service indexing",
            "--cluster hahn", "--stream-name {}".format(stream_name),
            "--event-time {}".format(event_time),
            "--output-tables-prefix //home/logfeller-dev/ovandriyanov/samples-indexing-{}".format(event_time)
        ])
        evaluation = {
            "create_quarantine_table_command": create_quarantine_table_command,
            "yt_stream_name": create_quarantine_table_command.split(" ")[6],
            "yt_cluster": create_quarantine_table_command.split(" ")[4],
        }
        eval_time = shlex.split(evaluation["create_quarantine_table_command"])[8]
        output_table_prefix = "//home/{}/logfeller/streams/".format(yt_account)
        output_table_prefix += evaluation["yt_stream_name"] + "/" + eval_time + "/stream"
        remote_temp_files_directory = "//home/{}/logfeller/yt_files_storage".format(yt_account)
        qm = QuarantineMaker(
            evaluation, output_table_prefix, remote_temp_files_directory,
            yt_token=YT_TOKEN, make_quarantine_bin_path=TEST_MAKE_QUARANTINE_PATH,
            logfeller_simple_yt_parser_bin_path=TEST_LOG_FELLER_SIMPLE_YT_PARSER
        )
        qm.create_quarantine_table()
        quarantine_table = qm.get_quarantine_table()
        print(quarantine_table)
    """

    def __init__(
        self, evaluation, output_table_prefix, remote_temp_files_directory,
        yt_token=YT_TOKEN, make_quarantine_bin_path=TEST_MAKE_QUARANTINE_PATH,
        logfeller_simple_yt_parser_bin_path=TEST_LOG_FELLER_SIMPLE_YT_PARSER,
        overwrite_event_time=None
    ):
        self._make_quarantine_bin_path = make_quarantine_bin_path
        self._logfeller_simple_yt_parser_bin_path = logfeller_simple_yt_parser_bin_path
        self._remote_temp_files_directory = remote_temp_files_directory
        self._output_table_prefix = output_table_prefix
        self._yt_token = yt_token
        self._evaluation = evaluation
        self._overwrite_event_time = overwrite_event_time

    def get_quarantine_table(self):
        """
        Eval path to yt table with quarantine sensitive data.
        """

        return self._output_table_prefix + ".quarantine"

    def _get_default_table(self):
        return self._output_table_prefix

    def _get_unparsed_table(self):
        return self._output_table_prefix + ".unparsed"

    def remove_non_quarantine_tables(self):
        import yt.wrapper as yt

        yt.config["token"] = self._yt_token
        yt.config["remote_temp_files_directory"] = self._remote_temp_files_directory
        yt.config.set_proxy(self._evaluation["yt_cluster"])

        path = self._get_default_table()
        try:
            yt.file_commands.remove(path)
        except yt.YtHttpResponseError:
            logging.info("QuarantineMaker:remove_non_quarantine_tables. Exception on deleting {}.".format(path))

        path = self._get_unparsed_table()
        try:
            yt.file_commands.remove(path)
        except yt.YtHttpResponseError:
            logging.info("QuarantineMaker:remove_non_quarantine_tables. Exception on deleting {}.".format(path))

    def _prepare_create_quarantine_cmd(self):
        """
        Get create_quarantine_table_command from evaluation and change output-table path
        """

        cmd = shlex.split(self._evaluation["create_quarantine_table_command"])

        # overwrite event_time
        if self._overwrite_event_time:
            cmd[8] = self._overwrite_event_time

        # overwrite make_quarantine bin path
        cmd[0] = self._make_quarantine_bin_path

        # overwrite output table path
        cmd[-1] = self._output_table_prefix

        # overwrite path to simple-parser-executable-path
        cmd.append("--simple-parser-executable-path")
        cmd.append(self._logfeller_simple_yt_parser_bin_path)

        return cmd

    def _prepare_env(self):
        """
        Reassign dir for temp files.
        Default //tmp not acceptable by robots and zombies.
        """

        env = os.environ.copy()
        env["YT_FILE_STORAGE"] = self._remote_temp_files_directory
        env["YT_TOKEN"] = self._yt_token
        return env

    def _exec_create_quarantine_table(self, cmd):
        sp.check_call(cmd, shell=False, env=self._prepare_env())

    def create_quarantine_table(self):
        cmd = self._prepare_create_quarantine_cmd()
        self._exec_create_quarantine_table(cmd)
