import json
import shlex
from sandbox.projects.security.logfeller_quarantine.utils import get_cur_timestamp


class Scheduler(object):
    """
    Scheduler select next evaluation to analyze across all evaluations.
    Also it has method store and read timings of previous analyse.
    """

    def __init__(self, yt_account, evaluations, reanalyze_period=None):
        self._yt_account = yt_account
        self._evaluations = evaluations
        self._reanalyze_period = reanalyze_period

    def _get_evaluation_time(self, evaluation):
        return shlex.split(evaluation["create_quarantine_table_command"])[8]

    def _build_streams_folder(self):
        return "//home/{}/logfeller/streams".format(self._yt_account)

    def _build_stream_folder(self, evaluation):
        return self._build_streams_folder() + "/" + evaluation["yt_stream_name"].replace("@", "-")

    def _build_meta_file_path(self, evaluation):
        return self._build_stream_folder(evaluation) + "/meta.json"

    def _build_quarantine_folder(self, evaluation):
        return self._build_stream_folder(evaluation) + "/" + self._get_evaluation_time(evaluation)

    def _build_quarantine_prefix(self, evaluation):
        return self._build_quarantine_folder(evaluation) + "/stream"

    def _get_meta_json_timings(self, evaluation):
        """
        Read timing meta file for corresponding evaluation.
        Return None if file does not exist.
        """
        import yt.wrapper as yt

        try:
            path = self._build_meta_file_path(evaluation)
            resp_stream = yt.file_commands.read_file(path)
            res = resp_stream.read()
            timings = json.loads(res)
            return timings
        except yt.YtHttpResponseError:
            return None

    def set_meta_last_start(self, evaluation, timestamp):
        """
        Set last_start value for timing of specified evaluation
        """
        import yt.wrapper as yt

        path = self._build_meta_file_path(evaluation)
        timings = self._get_meta_json_timings(evaluation)
        if not timings:
            timings = {
                "last_start": timestamp,
                "last_finish": None
            }
        else:
            timings["last_start"] = timestamp
        yt.file_commands.write_file(path, json.dumps(timings))

    def set_cur_meta_last_start(self, evaluation):
        self.set_meta_last_start(evaluation, get_cur_timestamp())

    def set_meta_last_finish(self, evaluation, timestamp):
        """
        Set last_finish value for timing of specified evaluation
        """
        import yt.wrapper as yt

        path = self._build_meta_file_path(evaluation)
        timings = self._get_meta_json_timings(evaluation)
        if not timings:
            timings = {
                "last_start": None,
                "last_finish": timestamp
            }
        else:
            timings["last_finish"] = timestamp
        yt.file_commands.write_file(path, json.dumps(timings))

    def set_cur_meta_last_finish(self, evaluation):
        self.set_meta_last_finish(evaluation, get_cur_timestamp())

    def _get_first_non_analysed_evaluation(self):
        """
        Returns
            1) Found evaluation without "last_start"
            2) List of pairs (evaluation, timings). To sort in future.
        """
        import yt.wrapper as yt

        eval_timing_list = list()
        for evaluation in self._evaluations:
            yt.config.set_proxy(evaluation["yt_cluster"])

            timings = self._get_meta_json_timings(evaluation)

            if timings and timings["last_start"]:
                eval_timing_list.append((evaluation, timings))
            else:
                return evaluation, None
        return None, eval_timing_list

    def _filter_time_passed(self, eval_timing_list):
        if not self._reanalyze_period:
            return False
        last_start_timestamp = eval_timing_list[1]["last_start"]
        current_timestamp = get_cur_timestamp()
        diff = current_timestamp - last_start_timestamp
        return diff > self._reanalyze_period

    def select_next_evaluation(self):
        """
        1) return first evaluation that was not analysed earlier
        2) return evaluation with the smallest "last_start" value
        """

        evaluation, eval_timing_list = self._get_first_non_analysed_evaluation()
        if evaluation:
            return evaluation

        week_passed_eval_timing_list = filter(self._filter_time_passed, eval_timing_list)
        if len(week_passed_eval_timing_list) == 0:
            return None

        sorted_week_passed_eval_timing_list = sorted(week_passed_eval_timing_list, key=lambda x: x[1]["last_start"])
        return sorted_week_passed_eval_timing_list[0][0]
