#!/usr/bin/env python

import calendar
import datetime
import os
import requests

#
# OAuth token. Get here:
#     https://wiki.yandex-team.ru/solomon/api/v2/#oauth.
# SWAGGER:
#     https://solomon.yandex-team.ru/swagger-ui.html.
#

SOLOMON_TOKEN = os.getenv("SOLOMON_TOKEN")


class SolomonAlertsServices(object):
    Indexing = "indexing"
    Ultrafast = "ultrafast"


class SolomonAlerts(object):
    """
    Example:
        ```
        sa = SolomonAlerts(token=SOLOMON_TOKEN)
        evaluations = sa.get_evaluations(cluster="hahn", services=[SolomonAlertsServices.Indexing])

        print("total: {}".format(len(evaluations)))
        print("first:")
        print(json.dumps(evaluations[0], indent=4, sort_keys=True))
        ```
    """

    BASE_API_URL = "http://solomon.yandex.net/api/v2"
    PROJECT_ID = "logfeller"
    ALERT_INDEXING_ID = "78b69d2e-d41e-4922-9878-e601ef7f620a"
    ALERT_ULTRAFAST_ID = "811a5fad-32fb-4923-b342-a211d4245afe"

    def __init__(self, token):
        self._AUTH_HEADER = {"Authorization": "OAuth {}".format(token)}

    @staticmethod
    def _str_to_datetime(str_time):
        return datetime.datetime.strptime(str_time, "%Y-%m-%dT%H:%M:%S.%fZ")

    @staticmethod
    def _dt_to_ts(dt):
        return calendar.timegm(dt.timetuple())

    @classmethod
    def _parse_subalert(cls, item):
        return {
            "subalert_id": item["id"],
            "alert_id": item["parentId"],
            "project_id": item["projectId"],
            "yt_cluster": item["labels"]["cluster"],
            "yt_stream_name": item["labels"].get("stream-name"),
            "yt_log_type": item["labels"].get("log-type"),
            "eval_timestamp": cls._dt_to_ts(cls._str_to_datetime(item["latestEval"])),
        }

    @classmethod
    def _parse_subalerts(cls, items):
        return list(map(lambda item: cls._parse_subalert(item), items))

    def _get_warn_subalerts(self, project_id, alert_id):
        """
        Iterate over subalerts with pagination.
        """

        url = self.BASE_API_URL + "/projects/{}/alerts/{}/subAlerts".format(project_id, alert_id)
        page_token = 0
        items = list()

        while True:

            print("[+] get_warn_subalerts. pageToken: {}.".format(page_token))

            params = {
                "filterByEvaluationStatus": "WARN",
                "pageSize": 100,
                "pageToken": page_token,
            }

            r = requests.get(url, params=params, headers=self._AUTH_HEADER)
            j = r.json()

            items += j["items"]
            next_page_token = j.get("nextPageToken")
            page_token = next_page_token

            if not page_token:
                break

        subalerts = self._parse_subalerts(items)
        return subalerts

    def _get_last_evaluation(self, project_id, alert_id, subalert):
        # url = BASE_API_URL + "/projects/{}/alerts/{}/subAlerts/{}/state/evaluation".format(project_id, alert_id, subalert["subalert_id"])
        url = "/".join([
            self.BASE_API_URL, "projects", project_id,
            "alerts", alert_id, "subAlerts", subalert["subalert_id"],
            "state", "evaluation"
        ])
        r = requests.get(url, headers=self._AUTH_HEADER)
        j = r.json()

        annotations = j["status"]["annotations"]

        evaluation = subalert.copy()
        evaluation["create_quarantine_table_command"] = annotations["Create quarantine table command"]
        evaluation["service"] = annotations["Service"]
        evaluation["quarantine_size"] = float(annotations["Quarantine size"])

        return evaluation

    def _get_last_evaluations(self, project_id, alert_id, subalerts):
        evaluations = list()
        for subalert in subalerts:
            evaluation = self._get_last_evaluation(project_id, alert_id, subalert)
            evaluations.append(evaluation)
        return evaluations

    def get_evaluations(self, cluster="hahn", services=[SolomonAlertsServices.Indexing]):
        """
        input:
            services: list one or more of SolomonAlertsServices

        """
        print("[+] 1/4. INDEXING. get_warn_subalerts.")
        indexing_subalerts = self._get_warn_subalerts(self.PROJECT_ID, self.ALERT_INDEXING_ID)
        print("[+] 2/4. INDEXING. get_last_evaluations.")
        indexing_evaluations = self._get_last_evaluations(self.PROJECT_ID, self.ALERT_INDEXING_ID, indexing_subalerts)
        print("[+] 3/4. ULTRAFAST. get_warn_subalerts.")
        ultrafast_subalerts = self._get_warn_subalerts(self.PROJECT_ID, self.ALERT_ULTRAFAST_ID)
        print("[+] 4/4. ULTRAFAST. get_last_evaluations.")
        ultrafast_evaluations = self._get_last_evaluations(self.PROJECT_ID, self.ALERT_ULTRAFAST_ID, ultrafast_subalerts)

        # merge indexing and ultrafast
        evaluations = indexing_evaluations + ultrafast_evaluations

        # only indexing right now
        evaluations = filter(lambda e: e["service"] in services, evaluations)

        # filter evaluations if cluster specified
        if cluster:
            evaluations = filter(lambda e: e["yt_cluster"] == cluster, evaluations)

        return list(evaluations)
