import requests


class YaResolveClient:
    """
    Use ya_resolve service to resolve
        - service
        - servicerole
        - department
        - person
        - wiki groups
        ...
        to responsible staff users.

    Usage:
        ya_resolve = YaResolveClient()
        r1 = ya_resolve.resolve_one("security")
        r2 = ya_resolve.resolve_many(["security", "procenkoeg"])
        r3 = ya_resolve.resolve_many_from_string("security,procenkoeg")
    """

    def __init__(self, api_url="https://ya-resolve.sec.yandex.net/api/v1/resolve"):
        self.api_url = api_url

    def _request(self, value):
        params = {"value": value}
        r = requests.get(self.api_url, params=params)
        try:
            json_data = r.json()
        except ValueError:
            return list()

        if json_data.get("status") != "ok":
            return list()
        return json_data.get("result")

    def _filter_only_success(self, results):
        return list(filter(lambda result: result.get("status") == "success", results))

    def _map_res_value_to_shortest_flow_depth(self, results):
        value_depth_dict = dict()
        for result in results:

            value = result["value"]
            flow_depth = len(result["flow"])

            if value not in value_depth_dict:
                value_depth_dict[value] = flow_depth

            elif flow_depth < value_depth_dict[value]:
                value_depth_dict[value] = flow_depth

        return value_depth_dict

    def _sort_res_values_by_flow_depth(self, value_depth_dict):
        items = value_depth_dict.items()
        sorted_items = sorted(items, key=lambda x: x[1])
        sorted_values = list(map(lambda x: x[0], sorted_items))
        return sorted_values

    def resolve_one(self, value, limit=10):
        if not value:
            return list()

        results = self._request(value)
        success_results = self._filter_only_success(results)
        value_depth_dict = self._map_res_value_to_shortest_flow_depth(success_results)
        values_sorted_by_flow_depth = self._sort_res_values_by_flow_depth(value_depth_dict)
        return values_sorted_by_flow_depth[:limit]

    def resolve_many(self, list_of_values, limit=10):
        value_flow_depth_pairs = list()

        if not list_of_values:
            return value_flow_depth_pairs

        for value in list_of_values:
            results = self._request(value)
            success_results = self._filter_only_success(results)
            value_depth_dict = self._map_res_value_to_shortest_flow_depth(success_results)
            value_flow_depth_pairs += value_depth_dict.items()

        sorted_items = sorted(value_flow_depth_pairs, key=lambda x: x[1])
        sorted_values = list(map(lambda x: x[0], sorted_items))

        resolved_values = list()
        for value in sorted_values:
            if value not in resolved_values:
                resolved_values.append(value)
                if len(resolved_values) == limit:
                    break

        return resolved_values

    def resolve_many_from_string(self, values_string, limit=10):
        if not values_string:
            return list()

        splitted = values_string.split(",")
        values = list(map(lambda x: x.strip(), splitted))
        resolved_values = self.resolve_many(values, limit=limit)

        return resolved_values
