import json
# import hashlib
import tempfile
import logging
import subprocess
from sandbox import sdk2
from sandbox.sandboxsdk.channel import channel
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.security.sec_check_quasar_image.common.utils import cached_property
from sandbox.projects.stop_leak.common import sectools
from sandbox.projects.stop_leak.common.yahec import HecSender


class SecCheckBasePlatformImage(sdk2.Task):
    """ Base checker class for different QUASAR platforms. """

    rootfs_dir = ''
    platform = ''   # Example: YANDEXSTATION_OTA_IMAGE, JBL_MUSIC_DEVICE_IMAGE, ...

    class Requirements(sdk2.Task.Requirements):
        container_resource = 1359925573
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        validate = sdk2.parameters.Bool('Validate found secrets', default=True)
        valid_only = sdk2.parameters.Bool('Report only validated secrets', default=False)
        send_to_splunk = sdk2.parameters.Bool('Send ant-secret results to splunk', default=True)
        suspend_after_image_unpack = sdk2.parameters.Bool('Suspend task after image unpack to get shell', default=False)

    def _prepare_env(self):
        """Prepare environment, install required packages, python modules, ..."""

        with sdk2.helpers.ProcessLog(self, logger="_prepare_env") as pl:
            cmd = "apt-get update && apt-get install -y unzip"
            sp.check_call(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT)

    @cached_property
    def temp_dir(self):
        return tempfile.mkdtemp()

    @cached_property
    def ant_path(self):
        """ Download ant-secret resource and return path to it. """

        return sectools.download_secret_search()

    @cached_property
    def ant_cmd(self):
        """ Prepare ant-secret command args. """

        _ant_cmd = [self.ant_path, '--status-code=0', '--format=json']
        if self.Parameters.validate:
            _ant_cmd.append("--validate")
        if self.Parameters.valid_only:
            _ant_cmd.append("--valid-only")
        return _ant_cmd

    def _run_ant_secret(self):
        """Run ant-secret on self.rootfs_dir folder and returns findings."""

        cmd = self.ant_cmd + [self.rootfs_dir]
        p = subprocess.Popen(cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE)
        res = p.communicate()

        logging.info("[+] ant-secret res: {}".format(res))
        logging.info("[+] ant-secret returncode: {}".format(p.returncode))

        stdout = res[0]

        json_res = json.loads(stdout)
        return json_res

    def _convert_ant_secret_results_to_splunk_format(self, results):
        """Process and convert ant secret results to splunk compatible format."""

        rootfs_prefix_length = len(self.rootfs_dir)

        splunk_results = list()
        for res in results:
            path = res.get("path", "")[rootfs_prefix_length:]

            for secret in res.get("secrets", list()):
                splunk_result = dict()
                splunk_result["path"] = path
                splunk_result["platform"] = self.platform
                splunk_result["validator"] = "ant_secret"
                splunk_result["line_no"] = secret.get("line_no")
                splunk_result["validated"] = secret.get("validated")
                splunk_result["secret_type"] = secret.get("type")

                additional = secret.get("additional")
                if additional:
                    splunk_result["sha1"] = additional.get("sha1")
                    del additional["sha1"]

                splunk_result["additional"] = additional

                splunk_results.append(splunk_result)

        return splunk_results

    def _send_results_to_splunk(self, splunk_results):
        """Process ant secret results and send them to splunk."""

        with HecSender(sdk2.Vault.data('SEC_CHECK_QUASAR_IMAGE_HEC_TOKEN')) as hec_sender:
            for splunk_result in splunk_results:
                hec_sender.send(**splunk_result)

    def _get_image_resource(self):
        """ Find last ota_image. Example:
        ```
        ota_image_resource = apihelpers.get_last_resource_with_attribute(
            self._get_ota_image_resource_type(), attribute_name='buildtype', attribute_value='release'
        )
        ```
        """
        raise NotImplementedError()

    @cached_property
    def ota_image(self):
        """Find required image sandbox resource and fetch it to "self.temp_dir/ota.zip" file."""

        ota_image_resource = self._get_image_resource()
        ota_image_resource_path = channel.task.sync_resource(ota_image_resource)

        with sdk2.helpers.ProcessLog(self, logger="ota_image") as pl:
            ota_image = "{}/ota.zip".format(self.temp_dir)
            cmd = "cp {} {}".format(ota_image_resource_path, ota_image)
            sp.check_call(cmd, shell=True, stdout=pl.stdout, stderr=sp.STDOUT)

        return ota_image

    def _unpack_image(self):
        """Unpack fetched image to some dir, set self.rootfs_dir value."""

        raise NotImplementedError()

    def _cleanup(self):
        """Routine to destruct allocated resources, for example umount images."""

        pass

    # def _check_certificates(self):
    #     config_file = self.rootfs_dir + "/system/vendor/quasar/quasar.cfg"
    #     try:
    #         f = open(config_file, "r")
    #         config = json.load(f)
    #         f.close()
    #     except IOError:
    #         return {"status": False, "reason": "FileNotFoundError"}
    #     except json.JSONDecodeError:
    #         return {"status": False, "reason": "JSONDecodeError"}
    #     except:
    #         return {"status": False, "reason": "Unknown Exception"}

    #     ca_cert_file = config.get("common", dict()).get("caCertsFile")
    #     if not ca_cert_file:
    #         return {"status": False, "reason": "caCertsFile not found in config file"}

    #     try:
    #         f = open(self.rootfs_dir + ca_cert_file, 'r')
    #         data = f.read().encode()
    #         f.close()
    #         s = hashlib.sha1()
    #         s.update(data.encode())
    #         if s.hexdigest() != "ee4988eee51491b949a2230c0fc4e1eae161fc9e":
    #             return {"status": False, "reason": "Hash mismatch"}
    #     except:
    #         return {"status": False, "reason": "Unknown Exception 2"}

    #     return {"status": True}

    def on_execute(self):
        """Main sanbox task excution task."""

        self._prepare_env()
        self._unpack_image()

        if self.Parameters.suspend_after_image_unpack:
            self.suspend()

        ant_secret_results = self._run_ant_secret()
        ant_secret_results = self._convert_ant_secret_results_to_splunk_format(ant_secret_results)

        # cert_check_result = self._check_certificates()
        # cert_check_result["validator"] = "certificates_checker"

        results = ant_secret_results  # + [cert_check_result]

        logging.info("[+] Splunk results: {}".format(json.dumps(results)))

        if self.Parameters.send_to_splunk:
            self._send_results_to_splunk(results)

        self._cleanup()
