import logging
import requests
import datetime
import json
import os.path
from os import listdir
from collections import defaultdict
from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import subprocess, ProcessLog
from sandbox.sandboxsdk import svn

URL = 'https://upload.stat.yandex-team.ru/_api/report/data/Yandex/Test/report/loc?scale='
SECRET_NAME, OWNER = 'OAUTH_TOKEN_STAT', 'bayan98'
RECOGNISED_LANGUAGES_LIST_FILE = 'recognised_languages.json'
CODE_DIR = 'web/report/lib/YxWeb'

RECOGNIZED_LANGUAGES = {'plh': 'Perl',
                        'plx': 'Perl',
                        'pm': 'Perl',
                        'perl': 'Perl'}


def read_recognised_languages(path):
    if not os.path.isfile(path):
        logging.info('send_loc_data: could not find recognised_languages.json at path {}'.format(path))
        return

    with open(path) as data_file:
        RECOGNIZED_LANGUAGES.update(json.load(data_file))
    logging.info('send_loc_data: RECOGNIZED_LANGUAGES: {}'.format(RECOGNIZED_LANGUAGES))


class SendLocDataReport(sdk2.Task):
    data = defaultdict(lambda: {'all': 0, 'blank': 0, 'code': 0, 'files': 0})

    class Parameters(sdk2.Parameters):
        arcadia_url = sdk2.parameters.ArcadiaUrl(required=True)

    def format_data(self):
        result_data = []
        for lang, num in self.data.iteritems():
            result_data.append({'all': num['all'],
                                'blank': num['blank'],
                                'code': num['code'],
                                'fielddate': datetime.datetime.now().strftime("%Y-%m-%d"),
                                'files': num['files'],
                                'language': lang,
                                'package': 'monthly'})
        return result_data

    def send_report(self, token, scale='daily'):
        logging.info('send_loc_data: Sending {} data'.format(scale))
        resp = requests.post(URL + scale,
                             headers={'Authorization': 'OAuth {}'.format(token)},
                             json={'data': self.format_data(), })
        logging.info('send_loc_data: Result of {} sending: {}'.format(scale, resp))

        if datetime.datetime.now().day == 1 and scale == 'daily':
            self.send_report(token, 'monthly')

    def count_lines_in_file(self, path):
        if '.' not in path:
            logging.info('send_loc_data: Unknown lagnuage format of file {}'.format(path))
            return

        format_name = path.split('.')[1]
        if format_name not in RECOGNIZED_LANGUAGES:
            logging.info('send_loc_data: Unknown lagnuage format of file {}'.format(path))
            return

        lang = RECOGNIZED_LANGUAGES[format_name]
        self.data[lang]['files'] += 1
        with open(path) as tempfile:
            for line in tempfile:
                self.data[lang]['all'] += 1
                if line.strip():
                    self.data[lang]['code'] += 1
                else:
                    self.data[lang]['blank'] += 1

    def count_lines_in_dir(self, directory):
        logging.info('send_loc_data: count lines in dir {}'.format(directory))
        for file in listdir(directory):
            if os.path.isfile(os.path.join(directory, file)):
                self.count_lines_in_file(os.path.join(directory, file))
            if os.path.isdir(os.path.join(directory, file)):
                self.count_lines_in_dir(os.path.join(directory, file))

    def on_execute(self):
        logging.info('send_loc_data: Start task')
        try:
            token = sdk2.Vault.data(OWNER, SECRET_NAME)
        except common.errors.VaultError:
            raise common.errors.TaskFailure("Unable to read token")

        arcadia_url = self.Parameters.arcadia_url
        logging.info('send_loc_data: arcadia url {}'.format(arcadia_url))
        arcadia_dir = svn.Arcadia.get_arcadia_src_dir(arcadia_url)
        logging.info('send_loc_data: arcadia_dir {}'.format(arcadia_dir))
        ya_path = os.path.join(arcadia_dir, 'ya')
        current_dir = os.path.join(arcadia_dir, 'sandbox', 'projects', 'send_loc_data', 'send_report')
        report_dir = os.path.join(arcadia_dir, CODE_DIR)

        read_recognised_languages(os.path.join(current_dir, RECOGNISED_LANGUAGES_LIST_FILE))
        self.execute('checkout', ya_path + ' make --checkout ' + CODE_DIR, cwd=arcadia_dir)
        logging.info('send_loc_data: report_dir: {}'.format(report_dir))

        self.count_lines_in_dir(report_dir)
        logging.info('send_loc_data: Data: {}'.format(self.format_data))
        self.send_report(token)

    def execute(self, logger, command, cwd='.', env=None):
        if env is None:
            env = {}
        env.update(os.environ)
        env['TZ'] = 'Europe/Moscow'
        with ProcessLog(self, logger=logger) as pl:
            retcode = subprocess.Popen(command.split(), stdout=pl.stdout, stderr=subprocess.STDOUT, cwd=cwd, env=env).wait()
            if retcode:
                common.errors.TaskFailure('%s failed' % logger)
            return
