# -*- coding: utf-8 -*-
import contextlib
import datetime
import logging
import os
import tarfile
import time

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
import sandbox.common.types.resource as ctr
import sandbox.projects.common.arcadia.sdk as arcadia_sdk
import sandbox.projects.common.nanny.nanny as nanny
import sandbox.projects.common.yappy.resources as ui_resources
import sandbox.projects.common.yappy.utils.node_manager as node_manager
import sandbox.sdk2 as sdk2

from sandbox.projects.setrace.resource_types import SETraceUIBundle


class BuildSETraceUI2(nanny.ReleaseToNannyTask2, sdk2.Task):
    """ Build Release Engine Binaries """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 30 * 60  # 30 min
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl("Url for arcadia", required=True)
        nodejs_archive = sdk2.parameters.Resource(
            "Node JS", resource_type=ui_resources.NodeJsArchive, state=(ctr.State.READY,), required=True
        )
        npm_build_command = sdk2.parameters.String("npm build command", default="build_production")
        ui_path = sdk2.parameters.String('Path to UI in arc', default='search/tools/setrace/ui')

    class Requirements(sdk2.task.Requirements):
        client_tags = ctc.Tag.GENERIC
        dns = ctm.DnsType.DNS64
        disk_space = 2 * 1024  # 2 Gb

    class Context(sdk2.Task.Context):
        sources_path = None
        dist_path = "dist"

    def on_execute(self):
        with arcadia_sdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as path:
            self.Context.sources_path = os.path.join(path, self.Parameters.ui_path)
            logging.debug('Directory %s: [%s]', self.Context.sources_path, os.listdir(self.Context.sources_path))
            self._setup_node()
            logging.debug('Start archiving UI package')
            with tarfile.TarFile('ui.tgz', mode='w') as tar:
                tar.add(
                    os.path.join(self.Context.sources_path, self.Context.dist_path),
                    self.Context.dist_path
                )
            sdk2.ResourceData(SETraceUIBundle(self, 'SETrace UI', 'ui.tgz')).ready()

    def _setup_node(self):
        logging.debug('Start setup of NodeJS')
        node = node_manager.NodeManager(self, 'http://npm.yandex-team.ru')
        node.setup(self.Context.sources_path, self.Parameters.nodejs_archive)

        logging.debug('Install node_modules to %s', self.Context.sources_path)
        node.install_modules(cwd=self.Context.sources_path)

        logging.debug('Start building UI')
        node.run_script(self.Parameters.npm_build_command, cwd=self.Context.sources_path)

    @contextlib.contextmanager
    def info_section(self, name):
        """ Copied from yappy/github """
        start = time.time()
        exception = None
        try:
            yield
        except Exception as e:
            exception = e
        finally:
            delta = datetime.timedelta(seconds=time.time() - start)
            delta -= datetime.timedelta(microseconds=delta.microseconds)

            state = 'DONE'
            message = '({}) {{}} {}'.format(delta, name)
            if exception:
                state = 'FAIL'
                message = '{}: {}: {}'.format(message, exception.__class__.__name__, exception)

            try:
                message = message.format(state)
            except Exception as e:
                message = 'info_section ({}) crashed (state={}): {}: {}'.format(name, state, e.__class__.__name__, e)

            self.set_info(message)

            if exception:
                raise exception
