# coding: utf-8

import os
import json
import time
import socket
import logging
import tarfile
import requests

from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry

from sandbox import sdk2
from sandbox.sdk2 import yav
from sandbox.sandboxsdk import process
from sandbox.sdk2.helpers import subprocess
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

import sandbox.projects.setrace.resource_types as srt
import sandbox.projects.resource_types as rt


class CheckSetraceRelease(sdk2.Task):
    class Context(sdk2.Task.Context):
        pass

    class Requirements(sdk2.Task.Requirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        SetraceBundle = sdk2.parameters.LastResource(
            "SETrace bundle",
            resource_type=srt.SETRACE_BUNDLE,
            required=True,
        )
        SetraceConfig = sdk2.parameters.LastResource(
            "SETrace config file",
            resource_type=srt.SETraceConfig,
        )
        Mongo = sdk2.parameters.Resource(
            'MongoDB package',
            resource_type=rt.MONGO_PACKAGE,
            required=True,
            default=911610993,
        )
        CheckOnlyJson = sdk2.parameters.Bool('Do not check UI', default=True)
        Request = sdk2.parameters.String('Request that will be traced', default='проверка')
        AliceKey = sdk2.parameters.String('Key for checking Alice trace', default='-62135769600')
        UseDefaultConfig = sdk2.parameters.Bool('Use defaut config', default=True)
        Verbose = sdk2.parameters.Bool('Check the box only if you want to debug', default=False)

    class Context(sdk2.Task.Context):
        pass

    def verbose(self, *args, **kwargs):
        if self.Parameters.Verbose:
            logging.debug(*args, **kwargs)
        else:
            pass

    def extract_tar(self, src, dst=None):
        self.verbose('Extracting tar files from "{src}" to "{dst}"'.format(src=src, dst=dst))
        if tarfile.is_tarfile(src):
            if not dst:
                idx = src.find('.tar')
                dst = src[:idx]
            tar = tarfile.open(src)
            tar.extractall(path=dst)
            tar.close()
            return dst

    def run_mongo(self):
        mongo_dir = sdk2.paths.make_folder("mongo", delete_content=True)
        mongo_path = str(sdk2.ResourceData(self.Parameters.Mongo).path)
        self.extract_tar(mongo_path, mongo_dir)
        try:
            mongo_args = '''--storageEngine wiredTiger --wiredTigerCacheSizeGB 1.0 --ipv6 --pidfilepath
                        ./state/mongo --logpath var/logs/mongo.log --logappend --logRotate rename
                        --dbpath var/lib/mongo --nounixsocket --replSet setrace --journal'''
            self.mongo_process = subprocess.Popen([mongo_dir + '/' + mongo_path.split('/')[-1][:-4] + '/bin/mongod', mongo_args],
                                                    shell=True)
        except subprocess.CalledProcessError as exc:
            logging.error("Running Mongo failed, exit code {code}.\n{msg}\n========\n".format(
                code=exc.returncode,
                msg=exc.output
            ))
            raise

    def run_setrace(self):
        setrace_dir = sdk2.paths.make_folder("setrace", delete_content=True)
        setrace_path = str(sdk2.ResourceData(self.Parameters.SetraceBundle).path)
        self.extract_tar(setrace_path, setrace_dir)
        setrace_binary_path = setrace_dir + '/setrace'
        try:
            if self.Parameters.UseDefaultConfig:
                # self.setrace_process = subprocess.Popen([setrace_binary_path], stderr=subprocess.PIPE, stdout=subprocess.PIPE, shell=True)
                self.setrace_process = subprocess.Popen([setrace_binary_path])
            else:
                setrace_config_path = str(sdk2.ResourceData(self.Parameters.SetraceConfig).path)
                self.setrace_process = subprocess.Popen([setrace_binary_path, '--cfg', setrace_config_path],
                                                        stderr=subprocess.PIPE, stdout=subprocess.PIPE, shell=True)
        except subprocess.CalledProcessError as exc:
            logging.error("Running SETrace failed, exit code {code}.\n{msg}\n========\n".format(
                code=exc.returncode,
                msg=exc.output
            ))
            raise

    def create_session(self):
        self.session = requests.Session()
        retries = Retry(total=10, backoff_factor=0.5, status_forcelist=[429, 500, 502, 503, 504])
        self.session.mount(self.Context.SEARCH_URL, HTTPAdapter(max_retries=retries))
        self.session.mount(self.Context.SETRACE_API_URL, HTTPAdapter(max_retries=retries))
        self.session.mount(self.Context.SETRACE_UI_URL, HTTPAdapter(max_retries=retries))

    def get_url(self, session, url, headers=None):
        try:
            if headers:
                session.headers.update(headers)
            r = session.get(url)
            r.raise_for_status()
        except requests.HTTPError as e:
            raise Exception("Failed to send request to url `{}`".format(url))
        except Exception as e:
            raise Exception('Failed to get url "{}"\nError: {}'.format(url, e))
        return r

    def make_request(self):
        request = self.get_url(self.session, self.Context.SEARCH_URL + self.Parameters.Request + '&dump=json')
        contents = request.text
        searched_str = 'Ya.SerpContext='
        context = contents[contents.find(searched_str)+len(searched_str):].split('<')[0]
        return json.loads(context)['reqid']

    def start_trace(self, token):
        request = self.get_url(self.session,
                                    self.Context.SETRACE_API_URL + '?reqid=' + self.Context.reqid,
                                    headers={'Authorization': 'OAuth %s' % self.Context.SETRACE_TOKEN})
        self.Context.uuid = request.json()['uuid']

    def wait_for_trace(self, token):
        finished = False
        while not finished:
            request = self.get_url(self.session, self.Context.SETRACE_API_URL + '/' + self.Context.reqid + '/status')
            trace_status = request.text
            self.verbose("%r", trace_status)
            finished = json.loads(trace_status)['is_finished']
            time.sleep(30)
        self.Context.status = json.loads(trace_status)

    def run_json_tests(self, token):
        try:
            request = self.get_url(self.session, self.Context.SETRACE_API_URL + '/' + self.Context.reqid + '/results')
            self.trace_results = request.json()
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to get trace result\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_API_URL + '/' + self.Context.reqid + '/results/tree')
            self.tree = request.json()
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to get trace tree\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_API_URL + '/' + self.Context.reqid + '/apphost_eventlog_json')
            self.apphost_eventlog = request.json()
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to get apphost eventlog\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_API_URL + '/' + self.Context.reqid + '/tracepage')
            self.tracepage = request.json()
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to get tracepage\nError: {}'.format(e))

        self.verbose("Trace result:\n%r\n", self.trace_results)
        self.verbose("Tree:\n%r\n", self.tree)
        assert len(self.trace_results["searchlist"]) > 1

    def run_ui_tests(self, token):
        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL)
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open main SETrace page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/search/' + self.Context.reqid + '/traceStatus')
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open trace status page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/search/' + self.Context.reqid + '/tracepage')
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open trace page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/search/' + self.Context.reqid + '/results/root')
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open root source page\nError: {}'.format(e))

        assert self.tree['searchId'] == 'root'
        some_source = self.tree['children'][0]
        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/search/' + self.Context.reqid + '/results/' +
                                    some_source['searchId'])
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open source `{}` page\nError: {}'.format(some_source['searchId'], e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/search/history')
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open history page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/admin/usage/status')
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open admin page\nError: {}'.format(e))

    def run_alice_tests(self, token):
        try:
            request = self.get_url(self.session, self.Context.SETRACE_UI_URL + '/alice/sessionsList?trace_by=' + self.Parameters.AliceKey)
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open Alice SETrace page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.HOSTNAME + ':18081/v1/alice?trace_by=' + self.Parameters.AliceKey)
            request.raise_for_status()
            last_trace = request.json()['uuids'][0]
        except Exception as e:
            raise Exception('Failed to open Alice API page\nError: {}'.format(e))

        try:
            request = self.get_url(self.session, self.Context.HOSTNAME + ':18081/ui/alice/traceByPrimaryKey/' + last_trace['url'][64:])
            request.raise_for_status()
        except Exception as e:
            raise Exception('Failed to open Alice trace page\nError: {}'.format(e))

    def setrace_communicate(self):
        time.sleep(30)
        stdout, stderr = self.setrace_process.communicate()
        self.verbose("SETrace stdout: %r", stdout)
        self.verbose("SETrace stderr: %r", stderr)
        self.verbose("SETrace returncode: %r", self.setrace_process.returncode)

    def check_connection(self):
        for url in [self.Context.SEARCH_URL, self.Context.SETRACE_UI_URL]:
            try:
                res = self.get_url(self.session, url)
                self.verbose('Checking connection to url `{}`'.format(url))
                self.verbose('curl result: {}'.format(res.text))
            except:
                self.verbose('Could not reach url `{}`'.format(url))

    def on_create(self):
        pass

    def on_save(self):
        pass

    def on_enqueue(self):
        host = 'localhost'
        self.Context.HOSTNAME = host
        self.Context.SETRACE_API_URL = host + ':18081/v1/search'
        self.Context.SETRACE_UI_URL = host + ':18081/ui'
        self.Context.SEARCH_URL = 'https://hamster.yandex.ru/yandsearch?text='

    def on_execute(self):
        self.setrace_token = sdk2.Vault.data('SETrace-HEC-token')
        self.staff_token = sdk2.Vault.data('OAUTH_STAFF')
        self.ydb_token = sdk2.Vault.data('OAUTH_YDB')

        os.environ['OAUTH_STAFF'] = self.staff_token
        os.environ['MONGO_PWD'] = ''
        os.environ['OAUTH_CLIENT_SECRET'] = '3f2839f0613a4edb9eda431071a30c75'
        os.environ['SOC_TOKEN'] = self.setrace_token
        os.environ['OAUTH_YDB'] = self.ydb_token

        self.run_mongo()
        self.verbose('Mongo is running!')

        self.run_setrace()
        self.verbose('SETrace is running!')

        # self.setrace_communicate()

        self.create_session()
        self.check_connection()

        self.Context.reqid = self.make_request()
        self.verbose('Request made! Reqid is: `{}`'.format(self.Context.reqid))

        self.start_trace(self.setrace_token)
        self.verbose('Trace started!')

        self.wait_for_trace(self.setrace_token)
        self.verbose('Trace finished.')

        self.run_json_tests(self.setrace_token)
        self.verbose('JSON tests ran successfully!')

        if not self.Parameters.CheckOnlyJson:
            self.run_ui_tests(self.setrace_token)
            self.verbose('UI tests ran successfully!')

            self.run_alice_tests(self.setrace_token)
            self.verbose('Alice tests ran successfully!')

        self.setrace_process.kill()
        self.setrace_process.wait()
        self.verbose('SETrace killed.')

        self.mongo_process.kill()
        self.mongo_process.wait()
        self.verbose('Mongo killed.')
