from distutils.dir_util import copy_tree
import logging
import os
import shutil
import tarfile

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
import sandbox.common.types.task as ctt


from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants


class ShinyDiscoveryModelsBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    ''' Task for building shard2.0 resource bundle for shiny_discovery '''

    class Parameters(sdk2.Task.Parameters):
        bundle_resource_type = sdk2.parameters.String(
            'Bundle resource type',
            default='SHINY_DISCOVERY_MODELS_BUNDLE',
            required=True
        )

        arcadia_root_url = sdk2.parameters.String(
            'Arcadia root url',
            required=False
        )

        with sdk2.parameters.Group('Models') as model_block:
            models_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with models_build_from_arcadia.value[True]:
                models_arcadia_path = sdk2.parameters.List(
                    'Arcadia paths',
                    required=True,
                )
                models_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=True
                )

    def do_copy(self, src, dst_dir="./"):
        real_dst = os.path.join(self.bundle_directory, dst_dir)
        if os.path.isdir(src):
            shutil.copytree(src, os.path.join(real_dst, os.path.basename(src)))
        else:
            shutil.copy2(src, os.path.join(real_dst, os.path.basename(src)))


    def do_build(self, target, revision, output_directory):
        with arcadiasdk.mount_arc_path("@".join((sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, revision))) as arcadia_dir:
            arcadiasdk.do_build(
                build_system=sdk_constants.DISTBUILD_BUILD_SYSTEM,
                source_root=arcadia_dir,
                targets=[target],
                results_dir=arcadia_dir,
                clear_build=False
            )
            copy_tree(os.path.join(arcadia_dir, target), output_directory)


    def check_params(self):
        logging.info('Checking params...')

    def create_bundle(self):
        logging.info('Start bundle building...')

        resource = sdk2.Resource[self.Parameters.bundle_resource_type](
            self, 'Shard2.0 bundle', 'resources')
        resource_data = sdk2.ResourceData(resource)
        self.bundle_directory = str(resource_data.path)

        os.mkdir(self.bundle_directory)

        # Add models
        if self.Parameters.models_build_from_arcadia:
            for path in self.Parameters.models_arcadia_path:
                self.do_build(path,  str(self.Parameters.models_svn_revision), os.path.join(self.bundle_directory, 'models'))

        resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.check_params:
            self.check_params()

        with self.memoize_stage.create_bundle:
            self.create_bundle()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)
