from distutils.dir_util import copy_tree
import logging
import os
import shutil
import tarfile

from sandbox import common
from sandbox import sdk2

from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.svn import Arcadia
import sandbox.common.types.task as ctt


from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.constants import constants as sdk_constants


class ShinyDiscoveryShard2ResourceBundle(nanny.ReleaseToNannyTask2, sdk2.Task):
    ''' Task for building shard2.0 resource bundle for shiny_discovery '''



    class Parameters(sdk2.Task.Parameters):
        bundle_resource_type = sdk2.parameters.String(
            'Bundle resource type',
            default='SHINY_DISCOVERY_DJ_SHARD_2_RESOURCE_BUNDLE',
            required=True
        )

        arcadia_root_url = sdk2.parameters.String(
            'Arcadia root url',
            required=False
        )

        with sdk2.parameters.Group('DSSM') as dssm_block:
            # models_resource = sdk2.parameters.Resource(
            #     'Ready resource',
            #     required=False
            # )

            dssm_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with dssm_build_from_arcadia.value[True]:
                dssm_arcadia_path = sdk2.parameters.List(
                    'Arcadia paths',
                    required=False,
                )
                dssm_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )
        

        with sdk2.parameters.Group('Models') as model_block:
            models_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with models_build_from_arcadia.value[True]:
                models_arcadia_path = sdk2.parameters.List(
                    'Arcadia paths',
                    required=False,
                )
                models_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )

        with sdk2.parameters.Group('RecommenderConfig') as recommender_config_block:
            recommender_config_resource = sdk2.parameters.Resource(
                'Ready resource',
                required=False
            )
            recommender_config_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with recommender_config_build_from_arcadia.value[True]:
                recommender_config_arcadia_path = sdk2.parameters.String(
                    'Arcadia path',
                    required=False
                )
                recommender_config_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )

        with sdk2.parameters.Group('ShardBuilderConfig') as shard_builder_config_block:
            shard_builder_config_resource = sdk2.parameters.Resource(
                'Ready resource',
                required=False
            )
            shard_builder_config_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with shard_builder_config_build_from_arcadia.value[True]:
                shard_builder_config_arcadia_path = sdk2.parameters.String(
                    'Arcadia path',
                    required=False
                )
                shard_builder_config_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )

        with sdk2.parameters.Group('AttributesCalcerConfig') as attributes_calcer_config_block:
            attributes_calcer_config_resource = sdk2.parameters.Resource(
                'Ready resource',
                required=False
            )
            attributes_calcer_config_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with attributes_calcer_config_build_from_arcadia.value[True]:
                attributes_calcer_config_arcadia_path = sdk2.parameters.String(
                    'Arcadia path',
                    required=False
                )
                attributes_calcer_config_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )

        with sdk2.parameters.Group('Custom resources') as custom_block:
            custom_resource = sdk2.parameters.Resource(
                'Ready resource',
                required=False,
                multiple=True
            )
            custom_build_from_arcadia = sdk2.parameters.Bool(
                'Build from arcadia now',
                required=False
            )
            with custom_build_from_arcadia.value[True]:
                custom_arcadia_path = sdk2.parameters.List(
                    'Arcadia paths',
                    required=False,
                )
                custom_svn_revision = sdk2.parameters.Integer(
                    'SVN Revision',
                    required=False
                )

    def do_copy(self, src, dst_dir="./"):
        real_dst = os.path.join(self.bundle_directory, dst_dir)
        if os.path.isdir(src):
            shutil.copytree(src, os.path.join(real_dst, os.path.basename(src)))
        else:
            shutil.copy2(src, os.path.join(real_dst, os.path.basename(src)))


    def do_build(self, target, revision, output_directory):
        with arcadiasdk.mount_arc_path("@".join((sdk2.svn.Arcadia.ARCADIA_TRUNK_URL, revision))) as arcadia_dir:
            arcadiasdk.do_build(
                build_system=sdk_constants.DISTBUILD_BUILD_SYSTEM,
                source_root=arcadia_dir,
                targets=[target],
                results_dir=arcadia_dir,
                clear_build=False
            )
            copy_tree(os.path.join(arcadia_dir, target), output_directory)


    def check_params(self):
        logging.info('Checking params...')
        params = self.Parameters

        if not params.recommender_config_resource and not params.recommender_config_arcadia_path:
            raise common.errors.TaskFailure('You should specify recommender.conf resource or arcadia path for building')

        if not params.shard_builder_config_resource and not params.shard_builder_config_arcadia_path:
            raise common.errors.TaskFailure('You should specify shard_builder.conf resource or arcadia path for building')

    def create_bundle(self):
        logging.info('Start bundle building...')

        resource = sdk2.Resource[self.Parameters.bundle_resource_type](
            self, 'Shard2.0 bundle', 'resources')
        resource_data = sdk2.ResourceData(resource)
        self.bundle_directory = str(resource_data.path)
        
        os.mkdir(self.bundle_directory)

        # Add models
        if self.Parameters.models_build_from_arcadia:
            for path in self.Parameters.models_arcadia_path:
                self.do_build(path,  str(self.Parameters.models_svn_revision), os.path.join(self.bundle_directory, 'models'))
        
        # Add dssm
        if self.Parameters.dssm_build_from_arcadia:
            for path in self.Parameters.dssm_arcadia_path:
                self.do_build(path,  str(self.Parameters.dssm_svn_revision), os.path.join(self.bundle_directory, 'processing_shard'))

        # Add recommender.conf
        if self.Parameters.recommender_config_build_from_arcadia:
            path = os.path.join(self.Parameters.arcadia_root_url, self.Parameters.recommender_config_arcadia_path)
            if self.Parameters.recommender_config_svn_revision:
                path += '@' + str(self.Parameters.recommender_config_svn_revision)
            Arcadia.export(path, os.path.join(self.bundle_directory, 'recommender.conf'))
        elif self.Parameters.recommender_config_resource:
            shutil.copyfile(str(sdk2.ResourceData(self.Parameters.recommender_config_resource).path), os.path.join(self.bundle_directory, 'recommender.conf'))

        # Add shard_builder.conf
        if self.Parameters.shard_builder_config_build_from_arcadia:
            path = os.path.join(self.Parameters.arcadia_root_url, self.Parameters.shard_builder_config_arcadia_path)
            if self.Parameters.shard_builder_config_svn_revision:
                path += '@' + str(self.Parameters.shard_builder_config_svn_revision)
            Arcadia.export(path, os.path.join(self.bundle_directory, 'shard_builder.conf'))
        elif self.Parameters.shard_builder_config_resource:
            shutil.copyfile(str(sdk2.ResourceData(self.Parameters.shard_builder_config_resource).path), os.path.join(self.bundle_directory, 'shard_builder.conf'))

        # Add attributes_calcer.conf
        if self.Parameters.attributes_calcer_config_build_from_arcadia:
            path = os.path.join(self.Parameters.arcadia_root_url, self.Parameters.attributes_calcer_config_arcadia_path)
            if self.Parameters.attributes_calcer_config_svn_revision:
                path += '@' + str(self.Parameters.attributes_calcer_config_svn_revision)
            Arcadia.export(path, os.path.join(self.bundle_directory, 'attributes_calcer.conf'))
        elif self.Parameters.attributes_calcer_config_resource:
            shutil.copyfile(str(sdk2.ResourceData(self.Parameters.attributes_calcer_config_resource).path), os.path.join(self.bundle_directory, 'attributes_calcer.conf'))

        # Add custom sandbox resources
        if self.Parameters.custom_resource:
            for f in self.Parameters.custom_resource:
                self.do_copy(str(sdk2.ResourceData(f).path))

        if self.Parameters.custom_build_from_arcadia:
            for path in self.Parameters.custom_arcadia_path:
                path = os.path.join(self.Parameters.arcadia_root_url, path)
                _, file_name = path.rstrip('/').rsplit('/', 1)
                if self.Parameters.custom_svn_revision:
                    path += '@' + str(self.Parameters.custom_svn_revision)
                Arcadia.export(path, os.path.join(self.bundle_directory, file_name))

        resource_data.ready()

    def on_execute(self):
        with self.memoize_stage.check_params:
            self.check_params()

        with self.memoize_stage.create_bundle:
            self.create_bundle()

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask2.on_release(self, additional_parameters)
        sdk2.Task.on_release(self, additional_parameters)