import os

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.fs import untar_archive
from sandbox.common.utils import Enum
from sandbox.common.types import resource
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk.environments import GCCEnvironment
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc


class SitesearchNodejsPackage(sdk2.Resource):
    """Node.js package for sitesearch tests"""
    releasable = True


class SitesearchIntegrationTestReport(sdk2.Resource):
    """
    Sitesearch integration test report
    """
    releasable = False


class TestStatus(Enum):
    DRAFT = None
    EXECUTING = None
    SUCCESS = None
    FAILURE = None


class SitesearchIntegrationTest(sdk2.Task):
    tests = ["e2e-yandex", "e2e-iframe", "e2e-htmlcss"]

    class Requirements(sdk2.Requirements):
        environments = [GCCEnvironment('5.3.0')]
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.LINUX_XENIAL

    class Parameters(sdk2.Task.Parameters):
        # git_reference = sdk2.parameters.String('git_reference', default='dev', description='Git checkout reference')
        serp_host = sdk2.parameters.String("SERP host for testing")

    class Context(sdk2.Task.Context):
        reports = {}

    def _execute_test(self, test_name):
        report = self.Context.reports[test_name]
        report['status'] = TestStatus.EXECUTING
        self.Context.save()
        with sdk2.helpers.ProcessLog(self, logger=test_name) as pl:
            os.environ['SERP_HOST'] = self.Parameters.serp_host
            process = sp.Popen([
                "npm", 'run', test_name], cwd=unicode(self.clone_path), stdout=pl.stdout, stderr=pl.stderr
            )
            process.timeout = 10800
            status_code = process.wait()
            resource = SitesearchIntegrationTestReport(
                self, "Report " + test_name, unicode(self.clone_path / "hermione-reports" / test_name)
            )
            sdk2.ResourceData(resource).ready()
            report['link'] = resource.http_proxy + "/index.html"
            if status_code == 0:
                report['status'] = TestStatus.SUCCESS
            else:
                report['status'] = TestStatus.FAILURE
            self.Context.save()
            return report

    def on_execute(self):
        for test_name in self.tests:
            self.Context.reports[test_name] = {
                'status': TestStatus.DRAFT,
                'link': None,
            }
        self.Context.save()
        self.clone_path = self.path('sitesearch')
        process = sp.Popen([
            "svn", "checkout",
            "svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/frontend/services/sitesearch2",
            unicode(self.clone_path),
        ])
        process.timeout = 5 * 60
        process.wait()

        node_resource = sdk2.Resource.find(
            SitesearchNodejsPackage,
            attrs={"version": "12.22.9"},
            state=resource.State.READY,
            order="-id"
        ).first()
        node_resource_data = sdk2.ResourceData(node_resource)
        node_resource_path = node_resource_data.path
        node_path = self.path("node")
        untar_archive(unicode(node_resource_path), unicode(node_path))
        with sdk2.helpers.ProcessLog(self, logger="node-bin") as pl:
            ps = sp.Popen(["ls", "-l", unicode(node_path)], stdout=pl.stdout)
            ps.wait()
        os.environ['PATH'] = unicode(node_path / 'bin') + ':' + os.environ['PATH']

        with sdk2.helpers.ProcessLog(self, logger="npm-install") as pl:
            process = sp.Popen([
                "npm", 'install', '-dd', '--registry=http://npm.yandex-team.ru'],
                cwd=unicode(self.clone_path),
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            process.timeout = 10 * 60
            retcode = process.wait()
            if retcode:
                raise sp.CalledProcessError(retcode, "npm install")

        success = True
        for test_name in self.tests:
            report = self._execute_test(test_name)
            if report['status'] == TestStatus.FAILURE:
                success = False
        if not success:
            raise TaskFailure("Some of test failed")

    @staticmethod
    def _render_status(status):
        return "<span class='status status_" + status.lower() + "'>" + status + "</a>"

    @staticmethod
    def _render_task_link(name, link):
        if link is None:
            return name
        else:
            return "<a class='link external' href='" + link + "' target='_blank'>" + name + "</a>"

    @property
    def footer(self):
        items = []
        for name, report in self.Context.reports.iteritems():
            link = report['link']
            status = report['status']
            items.append({"Test name": self._render_task_link(name, link), "Status": self._render_status(status)})
        return {
            "<h3>Test reports</h3>": items
        }
