# coding: utf-8

"""
Build gosky from arcadia/infra/gosky.

The result is single binary GOSKY_BINARY resource

gosky is the universal bootstrapper and updater for
skynet, which meant to be run via cron every few hours.
It uses static gcc and python compiled with that static
gcc. So, gosky should run on *any* posix platform, if linux
compatibility layer is turned on :).

As of now, gosky will run on any freebsd and any linux (both
amd64 and x86 variants).

Author: Vadim Fint <mocksoul@yandex-team.ru>
"""

import py
import os
import sys
import platform

from sandbox.common.types.client import Tag

import sandbox.sandboxsdk.svn as sdk_svn
import sandbox.sandboxsdk.parameters as sdk_parameters

from sandbox.projects import resource_types

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path


class CheckoutArcadiaFromURL(sdk_parameters.SandboxArcadiaUrlParameter):
    name = 'checkout_arcadia_from_url'
    description = 'SVN URL (arcadia)'
    default_value = sdk_svn.Arcadia.trunk_url() + '/infra/gosky'


class BuildGoSky(SandboxTask):
    type = 'BUILD_GOSKY'

    client_tags = Tag.Group.LINUX | Tag.Group.OSX

    input_parameters = [
        CheckoutArcadiaFromURL,
    ]

    execution_space = 3000

    def __checkout(self):
        url = self.ctx[CheckoutArcadiaFromURL.name]
        sdk_svn.Svn.checkout(
            sdk_svn.Arcadia.svn_url(url), 'svn'
        )
        self.svnpath = py.path.local('svn')

    def __build(self):
        self.distpath = py.path.local('.')
        self.buildpath = self.distpath.join('build')

        run_process([
                sys.executable, '-u',
                self.svnpath.join('doit').strpath,
                '--file', self.svnpath.join('dodo.py').strpath,
                '--dir', self.distpath.strpath,
                '--db-file', self.buildpath.join('.doit.db').strpath,
                'gosky',
            ],
            log_prefix='dodo',
        )

        platform = os.uname()[0].lower()
        self.gosky_binary = self.distpath.join('gosky.%s' % (platform, ))
        self.buildpath.join('gosky.%s' % (platform, )).move(self.gosky_binary)
        if platform == 'linux':
            self.gosky_uni_binary = self.distpath.join('gosky')
            self.buildpath.join('gosky').move(self.gosky_uni_binary)
        else:
            self.gosky_uni_binary = None

    def __cleanup(self):
        self.svnpath.remove()
        self.buildpath.remove()

    def __create_resources(self):
        attrs = {
            'platform': platform.platform(),
        }

        # Create universal resource BEFORE architecture depended
        # This is because BuildForAll task will sync only 1 resource with
        # max resource id.
        if self.gosky_uni_binary:
            self.create_resource(
                'gosky binary (%s)' % (self.descr, ),
                self.gosky_uni_binary.strpath,
                resource_types.GOSKY_BINARY,
                arch='any',
                attributes=attrs
            )

        self.create_resource(
            'gosky binary (%s)' % (self.descr, ),
            self.gosky_binary.strpath,
            resource_types.GOSKY_BINARY,
            arch=self.arch,
            attributes=attrs
        )

    def on_execute(self):
        remove_path('/home/sandbox/.local/lib/python2.6/site-packages/site.py')
        remove_path('/home/sandbox/.local/lib/python2.6/site-packages/site.pyc')
        self.__checkout()
        self.__build()
        self.__cleanup()
        self.__create_resources()

    def arcadia_info(self):
        # TODO: fix this weird stuff
        return None, True, None


__Task__ = BuildGoSky
