# -*- coding: utf-8 -*-

import logging
import os
import py
import sys


from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadia_sdk

from sandbox.projects.skynet import abstract_build_skycore_bundle as absb


class SkyboneBundle(absb.SkycoreBundle):
    """
        TGZ bundle with skybone inside and skycore service descriptor (skybone.scsd)
    """
    releasers = ['mocksoul', 'oplachkin', 'torkve', 'sempav', 'zeil', 'max7255', 'olegsenin', 'aculage']


class BuildSkyboneBundle(absb.BuildSkycoreBundle):
    """
        Build skybone (skynet copier (tm)) bundle.
        The result will be tgz bundle with skycore service descriptor inside.
    """
    sv_name = 'skybone'
    sv_namecap = 'Skybone'
    resource_type = SkyboneBundle

    class Parameters(sdk2.Task.Parameters):
        description = 'skybone bundle'

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            'SVN URL (arcadia)',
            default_value=sdk2.vcs.svn.Arcadia.ARCADIA_TRUNK_URL + '/infra/skybone'
        )
        version = sdk2.parameters.String(
            'Version to build',
            required=True
        )
        bb_token_vault_secret = sdk2.parameters.String(
            'bb_token_vault_secret',
            description='Sandbox Vault secret with bitbucket token',
            default='robot-netbeard-bb-token',
            required=False
        )

        with sdk2.parameters.Output:
            archive_name = sdk2.parameters.String('Target archive file name', required=True)
            service_resource = sdk2.parameters.Resource('Final resource', required=True)


    def _check_checkout(self, path):
        if not path.join('skybone.scsd').check(exists=1):
            raise Exception(
                'Checkout our sources does not looks like skybone sources '
                '(no skybone.scsd file inside %r (contents: %r))' % (
                    str(path), path.listdir()
                )
            )

    def _build(self):
        env = os.environ.copy()
        env.pop('LD_LIBRARY_PATH', None)  # overwise, svn will not work properly
        env['PYTHON'] = '/skynet/python/bin/python'
        env['VARIANT'] = self.sv_name
        env['BB_TOKEN'] = sdk2.Vault.data(None, self.Parameters.bb_token_vault_secret)

        source_path = py.path.local(self.source_path)
        cmd = source_path.join('build.sh').strpath, 'bundle'

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('build')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                cmd,
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
                cwd=self.source_path.strpath,
                env=env
            )
            proc.wait()
            assert proc.returncode == 0, 'Process %r exited with code %d' % (cmd, proc.returncode)


class BuildSkyboneBundleArc(BuildSkyboneBundle):

    class Parameters(sdk2.Task.Parameters):
        description = 'skybone bundle'

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            'Arcadia URL',
            default_value=sdk2.vcs.svn.Arcadia.ARCADIA_TRUNK_URL + '/infra/skybone'
        )
        use_arc_instead_of_aapi = sdk2.parameters.Bool('Use arc instead of aapi', default=True)

        version = sdk2.parameters.String(
            'Version to build',
            required=True
        )
        bb_token_vault_secret = sdk2.parameters.String(
            'bb_token_vault_secret',
            description='Sandbox Vault secret with bitbucket token',
            default='robot-netbeard-bb-token',
            required=False
        )
        arc_oauth_token = sdk2.parameters.YavSecret('Yav secret with arc OAuth token')

        with sdk2.parameters.Output:
            archive_name = sdk2.parameters.String('Target archive file name', required=True)
            service_resource = sdk2.parameters.Resource('Final resource', required=True)


    def _checkout_service(self):
        self.arc_mount = arcadia_sdk.mount_arc_path(
            self.Parameters.checkout_arcadia_from_url,
            use_arc_instead_of_aapi=self.Parameters.use_arc_instead_of_aapi,
            arc_oauth_token=self.Parameters.arc_oauth_token.data()[self.Parameters.arc_oauth_token.default_key]
        )
        checkout_path = self.arc_mount.__enter__()

        self.source_path = py.path.local(checkout_path).join('infra', 'skybone')
        self._check_checkout(self.source_path)

    def _cleanup(self):
        self.arc_mount.__exit__(None, None, None)
