# -*- coding: utf-8 -*-

import os
import py
import textwrap

import requests
import lxml.etree

from sandbox.projects import resource_types

import sandbox.common.types.client as ctc
from sandbox.sandboxsdk import parameters, task, process, svn, channel


class BuildSkynet(task.SandboxTask):
    """
        Build skynet package and binary.
        The result will be package (run.py) and new type binary (skynet.bin).

        This task uses BuildSkynetDeps resources (pre-build platform-specific python+libraries).

        Author: Vadim Fint <mocksoul@yandex-team.ru>
    """
    type = 'BUILD_SKYNET'
    release_to = ['skynet@yandex-team.ru']
    client_tags = ctc.Tag.LINUX_TRUSTY

    class CheckoutArcadiaFromURL(parameters.SandboxSvnUrlParameter):
        name = 'checkout_arcadia_from_url'
        description = 'SVN URL (arcadia)'
        default_value = svn.Arcadia.trunk_url()

    class SkynetVersion(parameters.SandboxStringParameter):
        name = 'skynet_version'
        description = 'Version to build'

    class ZhirnayaZapis(parameters.SandboxInfoParameter):
        name = 'zhirnaya_zapis'
        description = 'Building options'

    class Debug(parameters.SandboxBoolParameter):
        name = 'debug'
        description = 'Run with debug info in logs'

    input_parameters = [
        CheckoutArcadiaFromURL,
        SkynetVersion,
        ZhirnayaZapis,
        Debug,
    ]

    execution_space = 1000

    def __checkout_skynet(self):
        checkout_path = 'arc'
        self.svnpath = py.path.local(checkout_path)

        parsed_url = svn.Arcadia.parse_url(self.ctx[self.CheckoutArcadiaFromURL.name])
        svn.Arcadia.checkout(url=self.ctx[self.CheckoutArcadiaFromURL.name], path=checkout_path, depth='empty')
        svn.Arcadia.update(path='%s/skynet' % (checkout_path, ), revision=parsed_url.revision, set_depth='infinity')

    def __build(self):
        self.buildpath = py.path.local('build')

        env = os.environ.copy()
        env.pop('LD_LIBRARY_PATH', None)  # overwise, svn will not work properly

        cmd = [self.svnpath.join('skynet', 'packages', 'make.sh').strpath]

        if self.ctx['skynet_version']:
            cmd.extend(['--version', self.ctx['skynet_version']])

        url_parts = self.ctx[self.CheckoutArcadiaFromURL.name].rsplit('@', 1)

        cmd.extend(['--svnurl', url_parts[0]])
        if len(url_parts) > 1:
            cmd.extend(['--svnrevision', url_parts[1]])

        if self.ctx['debug']:
            cmd = ['bash', '-x'] + cmd

        process.run_process(
            cmd,
            log_prefix='make.sh',
            environment=env,
        )

        skynetbin = self.buildpath.join('skynet.bin')

        # Make ./dist folder and move binaries there
        self.distdir = py.path.local('dist')
        if self.distdir.check(exists=1):
            self.distdir.remove()
        self.distdir.ensure(dir=1)

        skynetbin.move(self.distdir.join(skynetbin.basename))

        out = process.run_process(
            [self.distdir.join(skynetbin.basename).strpath, '--no-motd', '--version'],
            outs_to_pipe=True,
            timeout=60,
        ).stdout.read().strip().split('\n')[0]

        assert 'Version: ' in out, 'Failed to grab builded skynet.bin version!'
        version = out.split(': ', 1)[1]

        if self.ctx['skynet_version']:
            assert self.ctx['skynet_version'] == version, 'Builded skynet.bin version does not the same as requested!'
        else:
            self.ctx['skynet_version'] = version

    def __cleanup(self):
        self.svnpath.remove()
        self.buildpath.remove()

    def __create_resources(self):
        skynet_bin_resource = self.create_resource(
            'skynet.bin (%s)' % (self.descr, ),
            self.path('dist/skynet.bin'),
            resource_types.SKYNET_BINARY,
            arch='any',
            attributes={
                'version': self.ctx['skynet_version'],
                'svn_url': self.ctx[self.CheckoutArcadiaFromURL.name],
            },
        )

        self.ctx['skynet_bin_resource_id'] = skynet_bin_resource.id

    def on_execute(self):
        self.__checkout_skynet()
        self.__build()
        self.__cleanup()
        self.__create_resources()

    def mds_upload(self, path, store_path):
        base_uri = 'http://storage-int.mds.yandex.net:1111'
        get_uri_http = 'http://storage-int.mds.yandex.net/get-skynet'
        get_uri_https = 'https://storage-int.mds.yandex.net/get-skynet'

        auth = self.get_vault_data('SKYNET', 'skynet_mds_auth')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with open(path, 'rb') as fp:
            response = requests.get(base_uri + '/upload-skynet/' + store_path, headers=headers, data=fp)

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return {
                    'http': get_uri_http + '/' + uploaded_key,
                    'https': get_uri_https + '/' + uploaded_key,
                }, uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key, ))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')
        else:
            raise Exception('Unable to upload to mds: code %d' % (response.status_code, ))

    def on_release(self, additional_parameters):
        resource_id = self.ctx['skynet_bin_resource_id']
        resource_path = self.sync_resource(resource_id)

        info = self.mds_upload(
            resource_path,
            self.ctx['skynet_version'] + '/' + 'skynet.bin'
        )
        if not info:
            raise Exception('Unable to upload to MDS')

        mds_uri, mds_key = info

        channel.channel.sandbox.set_resource_attribute(resource_id, 'mds_key_skynetbin', 'skynet:' + mds_key)

        for typ, uri in mds_uri.items():
            channel.channel.sandbox.set_resource_attribute(
                resource_id, '%s_mds' % (typ, ), uri
            )
            channel.channel.sandbox.set_resource_attribute(
                resource_id, '%s_mds_weight' % (typ, ), 50 if typ == 'http' else 51
            )

        for scheme in 'http', 'https':
            channel.channel.sandbox.set_resource_attribute(
                resource_id,
                scheme + '_sandbox_proxy',
                scheme + '://proxy.sandbox.yandex-team.ru/%d' % (resource_id, )
            )
            channel.channel.sandbox.set_resource_attribute(
                resource_id,
                scheme + '_sandbox_proxy_weight',
                10 if scheme == 'http' else 11
            )

        return task.SandboxTask.on_release(self, additional_parameters)

    @property
    def release_template(self):
        revision, tag, branch = self.arcadia_info()

        if self.ctx['skynet_version']:
            subject = 'skynet v' + self.ctx['skynet_version']
        elif tag:
            subject = 'skynet v' + tag
        else:
            subject = 'skynet v' + branch + '@' + revision

        return self.ReleaseTemplate(
            ['torkve@yandex-team.ru'],
            subject,
            textwrap.dedent('''\
                service1:

                  ** Bugs
                     * SKYDEV-XXX: bug #1

                  ** Features
                     * SKYDEV-YYY: feature #1

                service2:

                  ** Features
                     * SKYDEV-YYY: feature #1
            '''),
            ['stable']
        )

    def arcadia_info(self):
        """ Return revision, tagname and branchname for release form. """
        parsed_url = svn.Arcadia.parse_url(self.ctx[self.CheckoutArcadiaFromURL.name])
        path = parsed_url.path
        revision = parsed_url.revision
        if path.startswith('arc/branches/skynet'):
            branch = path.split('/')[3]
        else:
            branch = None

        if path.startswith('arc/tags/skynet'):
            tag = path.split('/')[3]
        else:
            tag = None

        return revision, tag, branch


__Task__ = BuildSkynet
