# -*- coding: utf-8 -*-
"""
Build skynet dependencies resource.
The result will consist of several shared pre-build
tarballs shared via Skynet.Copier.

Building is made using SSH on servers.

After that these resources used in BuildSkynet task.

Author: Vadim Fint <mocksoul@yandex-team.ru>
"""

import os

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag

from sandbox.projects import resource_types

from sandbox.sandboxsdk import parameters, ssh
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder, get_unique_file_name, get_logs_folder
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.projects.common.arcadia import sdk as arcadia_sdk
from sandbox.projects.common.vcs.arc import Arc


class BuildSkynetDeps(SandboxTask):
    type = 'BUILD_SKYNET_DEPS'

    client_tags = (Tag.GENERIC | Tag.SKYNET) & Tag.Group.LINUX

    class CheckoutArcadiaFromURL(parameters.SandboxArcadiaUrlParameter):
        name = 'checkout_arcadia_from_url'
        description = 'SVN URL (arcadia)'
        default_value = Arcadia.trunk_url()

    class BuilderHosts(parameters.SandboxStringParameter):
        name = 'builder_hosts'
        description = 'Builder hosts'
        default_value = 'localhost ovid.search.yandex.net ymake-dev-mac2.search.yandex.net'

    input_parameters = [
        CheckoutArcadiaFromURL,
        BuilderHosts,
    ]

    execution_space = 200000
    dns = ctm.DnsType.DNS64

    def checkout(self):
        """ Step: Checkout arcadia/skynet/packages """

        url = self.ctx[self.CheckoutArcadiaFromURL.name]
        return arcadia_sdk.mount_arc_path(url, use_arc_instead_of_aapi=True)

    def run_builder(self, path):
        """ Step: run build.py with appropriate options """

        task_logs_folder = get_logs_folder()
        build_log_fp = open(get_unique_file_name(task_logs_folder, 'buildlog.htm'), 'w')
        build_log_err_fp = open(get_unique_file_name(task_logs_folder, 'buildlog.err'), 'w')

        self.set_info('Executing builder...')
        with ssh.Key(self, 'SKYNET', 'robot_skybot_key'):
            try:
                curdir = os.getcwd()
                os.chdir(os.path.join(path, 'skynet/packages'))
                self._subprocess_ex(
                    [
                        './build.py',
                        '--logdir', task_logs_folder,
                        '--remote-run', self.ctx['builder_hosts'],
                        '--output', os.path.join(curdir, 'skynet-deps'),
                        '--html',
                    ],
                    wait=True,
                    wait_timeout=7200,
                    check=True,
                    extra_env={},
                    stdout=build_log_fp,
                    stderr=build_log_err_fp,
                    stdin=None
                )
            except Exception as ex:
                raise SandboxTaskFailureError('Build failed: %s' % (ex, ))

        self.set_info('Build completed')

    def on_execute(self):
        arc_path = os.path.dirname(Arc().binary_path)
        os.environ['PATH'] = os.getenv('PATH', '') + ':' + arc_path

        with self.checkout() as path:
            self.run_builder(path)

        self.create_resource(
            'skynet-deps (%s)' % (self.descr, ),
            self.path('skynet-deps'),
            resource_types.SKYNET_DEPS,
            arch='any',
        )


__Task__ = BuildSkynetDeps
