# -*- coding: utf-8 -*-

import os
import py
import sys
import textwrap

import requests
import lxml.etree

from sandbox.projects import resource_types

from sandbox.sandboxsdk import parameters, task, process, svn, channel


class BuildSkynetService(task.SandboxTask):
    """
        Build common skynet service.
        The result will be package (run.py) and new type binary (skynet.bin).
    """
    type = 'BUILD_SKYNET_SERVICE'
    release_to = ['skynet@yandex-team.ru']

    class CheckoutArcadiaFromURL(parameters.SandboxSvnUrlParameter):
        name = 'checkout_arcadia_from_url'
        description = 'SVN URL (arcadia)'
        default_value = svn.Arcadia.trunk_url()

    class ServiceName(parameters.SandboxStringParameter):
        name = "service_name"
        description = "Service to build"
        required = True

    class ServiceVersion(parameters.SandboxStringParameter):
        name = 'service_version'
        description = 'Version to build'
        required = True

    input_parameters = [
        CheckoutArcadiaFromURL,
        ServiceName,
        ServiceVersion,
    ]

    execution_space = 1000

    def __checkout_service(self):
        checkout_path = 'arc'
        self.svnpath = py.path.local(checkout_path)

        parsed_url = svn.Arcadia.parse_url(self.ctx[self.CheckoutArcadiaFromURL.name])
        svn.Arcadia.checkout(url=self.ctx[self.CheckoutArcadiaFromURL.name], path=checkout_path, depth='empty')
        svn.Arcadia.update(path='%s/skynet' % (checkout_path, ), revision=parsed_url.revision, set_depth='infinity')

        self.service_path = self.svnpath.join('skynet', 'services', self.ctx['service_name']).strpath
        if os.path.exists(self.service_path):
            pass
        elif os.path.exists(self.service_path + '.link'):
            service_url = open(self.service_path + '.link').read().strip()
            svn.Arcadia.checkout(url=service_url, path=self.service_path, depth='infinity')
        else:
            raise Exception("Service cannot be found in skynet")

    def __build(self):
        env = os.environ.copy()
        env.pop('LD_LIBRARY_PATH', None)  # overwise, svn will not work properly
        env['PYTHON'] = sys.executable

        service_path = py.path.local(self.service_path)
        if service_path.join('build.py').check(exists=1):
            cmd = [sys.executable, 'build.py', 'bundle']
        elif service_path.join('build.sh').check(exists=1):
            cmd = [service_path.join('build.sh').strpath, 'bundle']
        else:
            raise Exception("Don't know how to build %s" % self.ctx['service_name'])

        process.run_process(
            cmd,
            log_prefix='build',
            environment=env,
            work_dir=self.service_path,
        )

    def __cleanup(self):
        self.svnpath.remove()

    def __create_resources(self):
        self.ctx['archive_name'] = '%s-%s.tgz' % (self.ctx['service_name'], self.ctx['service_version'])
        snapshot_path = py.path.local(self.service_path).join('snapshot').strpath
        process.run_process(
            ['tar', '-C', snapshot_path, '-czvf', self.ctx['archive_name'], '.'],
            log_prefix='archive',
        )

        resource = self.create_resource(
            '%s (%s)' % (self.ctx['archive_name'], self.descr,),
            self.path(self.ctx['archive_name']),
            resource_types.SKYCORE_SERVICE,
            arch='any',
            attributes={
                'version': self.ctx['service_version'],
                'svn_url': self.ctx[self.CheckoutArcadiaFromURL.name],
            },
        )

        self.ctx['service_resource_id'] = resource.id

    def on_execute(self):
        self.__checkout_service()
        self.__build()
        self.__create_resources()
        self.__cleanup()

    def mds_upload(self, path, store_path):
        base_uri = 'http://storage-int.mds.yandex.net:1111'
        get_uri_http = 'http://storage-int.mds.yandex.net/get-skynet'
        get_uri_https = 'https://storage-int.mds.yandex.net/get-skynet'

        auth = self.get_vault_data('SKYNET', 'skynet_mds_auth')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with open(path, 'rb') as fp:
            response = requests.get(base_uri + '/upload-skynet/' + store_path, headers=headers, data=fp)

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return {
                    'http': get_uri_http + '/' + uploaded_key,
                    'https': get_uri_https + '/' + uploaded_key,
                }, uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key, ))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')
        else:
            raise Exception('Unable to upload to mds: code %d' % (response.status_code, ))

    def on_release(self, additional_parameters):
        resource_id = self.ctx['service_resource_id']
        resource_path = self.sync_resource(resource_id)

        info = self.mds_upload(
            resource_path,
            self.ctx['service_name'] + '/' + self.ctx['archive_name']
        )
        if not info:
            raise Exception('Unable to upload to MDS')

        mds_uri, mds_key = info

        channel.channel.sandbox.set_resource_attribute(resource_id, 'mds_key_service', 'skynet:' + mds_key)

        for typ, uri in mds_uri.items():
            channel.channel.sandbox.set_resource_attribute(
                resource_id, '%s_mds' % (typ, ), uri
            )
            channel.channel.sandbox.set_resource_attribute(
                resource_id, '%s_mds_weight' % (typ, ), 50 if typ == 'http' else 51
            )

        for scheme in 'http', 'https':
            channel.channel.sandbox.set_resource_attribute(
                resource_id,
                scheme + '_sandbox_proxy',
                scheme + '://proxy.sandbox.yandex-team.ru/%d' % (resource_id, )
            )
            channel.channel.sandbox.set_resource_attribute(
                resource_id,
                scheme + '_sandbox_proxy_weight',
                10 if scheme == 'http' else 11
            )

        task.SandboxTask.on_release(self, additional_parameters)

    @property
    def release_template(self):
        revision, tag, branch = self.arcadia_info()

        if self.ctx['service_version']:
            subject = '%s v%s' % (self.ctx['service_name'], self.ctx['service_version'])
        elif tag:
            subject = '%s v%s' % (self.ctx['service_name'], tag)
        else:
            subject = '%s v%s@%s' % (self.ctx['service_name'], branch, revision)

        return self.ReleaseTemplate(
            ['skynet@yandex-team.ru'],
            subject,
            textwrap.dedent('''\
                Changelog:

                  ** Bugs
                     * SKYDEV-XXX: bug #1

                  ** Features
                     * SKYDEV-YYY: feature #1
            '''),
            ['stable']
        )

    def arcadia_info(self):
        """ Return revision, tagname and branchname for release form. """
        parsed_url = svn.Arcadia.parse_url(self.ctx[self.CheckoutArcadiaFromURL.name])
        path = parsed_url.path
        revision = parsed_url.revision
        if path.startswith('arc/branches/skynet'):
            branch = path.split('/')[3]
        else:
            branch = None

        if path.startswith('arc/tags/skynet'):
            tag = path.split('/')[3]
        else:
            tag = None

        return revision, tag, branch


__Task__ = BuildSkynetService
