# -*- coding: utf-8 -*-

import logging
import py
import textwrap

import requests
import lxml.etree

from sandbox import sdk2


class SkycoreBundle(sdk2.resource.Resource):
    """
        TGZ bundle with service inside and skycore service descriptor (service.scsd).
    """

    releasable = True
    any_arch = True
    executable = False
    auto_backup = True
    releasers = ['SKYNET']

    # custom attributes
    version = sdk2.Attributes.String("Bundle version", required=True, default="0.0")
    svn_url = sdk2.Attributes.String("SVN URL", required=False, default="trunk")

    # mds params, filled on release
    http_mds = sdk2.Attributes.String('HTTP mds url')
    http_mds_weight = sdk2.Attributes.Integer('HTTP mds url weight')
    http_sandbox_proxy = sdk2.Attributes.String('HTTP sandbox proxy url')
    http_sandbox_proxy_weight = sdk2.Attributes.Integer('HTTP sandbox proxy url weight')
    https_mds = sdk2.Attributes.String('HTTPS mds url')
    https_mds_weight = sdk2.Attributes.Integer('HTTPS mds url weight')
    https_sandbox_proxy = sdk2.Attributes.String('HTTPS sandbox proxy url')
    https_sandbox_proxy_weight = sdk2.Attributes.Integer('HTTPS sandbox proxy url weight')
    mds_key_service = sdk2.Attributes.String('MDS skynet key (service:key)')


def make_parameters(name):
    class _params(sdk2.Task.Parameters):
        description = '%s bundle' % (name, )

        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
            'SVN URL (arcadia)',
            default_value=sdk2.vcs.svn.Arcadia.ARCADIA_TRUNK_URL + '/infra/%s' % (name, )
        )
        version = sdk2.parameters.String(
            'Version to build',
            required=True
        )
        with sdk2.parameters.Output:
            archive_name = sdk2.parameters.String('Target archive file name', required=True)
            service_resource = sdk2.parameters.Resource('Final resource', required=True)

    return _params


class BuildSkycoreBundle(sdk2.Task):
    """
        Build skybone (skynet copier (tm)) bundle.
        The result will be tgz bundle with skycore service descriptor inside.
    """
    sv_name = 'my_service'
    sv_namecap = 'MyService'
    release_to = ['skynet@yandex-team.ru']
    resource_type = SkycoreBundle

    Parameters = make_parameters(sv_name)

    class Requirements(sdk2.Task.Requirements):
        disk_space = 1000

    def _check_checkout(self, path):
        return

    def _checkout_service(self):
        checkout_path = 'src'
        self.svnpath = py.path.local(checkout_path)

        sdk2.vcs.svn.Arcadia.checkout(
            url=self.Parameters.checkout_arcadia_from_url,
            path=checkout_path,
            depth=sdk2.vcs.svn.Arcadia.Depth.INFINITY
        )

        self.source_path = self.svnpath  # /src
        self._check_checkout(self.source_path)

    def _build(self):
        raise NotImplementedError('build is not implemented')

    def _cleanup(self):
        self.svnpath.remove()

    def _create_resources(self):
        snapshot_path = py.path.local(self.source_path).join('snapshot').strpath

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('archive')) as pl:
            proc = sdk2.helpers.subprocess.Popen(
                ['tar', '-C', snapshot_path, '-czvf', self.Parameters.archive_name, '.'],
                stdout=pl.stdout,
                stderr=sdk2.helpers.subprocess.STDOUT,
            )
            proc.wait()
            assert proc.returncode == 0, 'Unable to create tgz bundle: tar exited with code %d' % (proc.returncode, )

        self.Parameters.service_resource = self.resource_type(
            self,
            '%s (%s)' % (self.Parameters.archive_name, self.Parameters.description),
            self.Parameters.archive_name,
            version=self.Parameters.version,
            svn_url=self.Parameters.checkout_arcadia_from_url,
        )

    def on_save(self):
        version = self.Parameters.version
        if '.' in version and version.count('.') == 1:
            # 1.2 => 1.2.0
            version += '.0'
            self.Parameters.version = version
        self.Parameters.description = '%s bundle v%s' % (self.sv_name, self.Parameters.version, )

        tags = set(tag.lower() for tag in self.Parameters.tags)

        if 'skynet' not in tags:
            self.Parameters.tags.append('skynet')

        if 'skycore-bundle' not in tags:
            self.Parameters.tags.append('skycore-bundle')

        if self.sv_name not in tags:
            self.Parameters.tags.append(self.sv_name)

    def on_execute(self):
        if not self.Parameters.archive_name:
            self.Parameters.archive_name = '%s-%s.tgz' % (self.sv_name, self.Parameters.version, )

        self._checkout_service()
        self._build()
        self._create_resources()
        self._cleanup()

    def mds_upload(self, path, store_path):
        base_uri = 'http://storage-int.mds.yandex.net:1111'
        get_uri_http = 'http://storage-int.mds.yandex.net/get-skynet'
        get_uri_https = 'https://storage-int.mds.yandex.net/get-skynet'

        auth = sdk2.Vault.data('SKYNET', 'skynet_mds_auth')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with path.open('rb') as fp:
            response = requests.get(base_uri + '/upload-skynet/' + store_path, headers=headers, data=fp)

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return {
                    'http': get_uri_http + '/' + uploaded_key,
                    'https': get_uri_https + '/' + uploaded_key,
                }, uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key, ))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')
        else:
            raise Exception('Unable to upload to mds: code %d' % (response.status_code, ))

    def on_release(self, additional_parameters):
        resource = self.Parameters.service_resource
        resource_path = sdk2.ResourceData(resource).path

        info = self.mds_upload(
            resource_path,
            self.sv_name + '/' + self.Parameters.archive_name
        )
        if not info:
            raise Exception('Unable to upload to MDS')

        mds_uri, mds_key = info

        resource.mds_key_service = 'skynet:' + mds_key

        for typ, uri in mds_uri.items():
            setattr(resource, '%s_mds' % (typ, ), uri)
            setattr(resource, '%s_mds_weight' % (typ, ), 50 if typ == 'http' else 51)

        for scheme in 'http', 'https':
            setattr(
                resource,
                scheme + '_sandbox_proxy',
                scheme + '://proxy.sandbox.yandex-team.ru/%d' % (resource.id, )
            )
            setattr(
                resource,
                scheme + '_sandbox_proxy_weight',
                10 if scheme == 'http' else 11
            )

        super(BuildSkycoreBundle, self).on_release(additional_parameters)

    @property
    def release_template(self):
        revision, tag, branch = self.arcadia_info()

        if self.Parameters.version:
            subject = '%s v%s' % (self.sv_namecap, self.Parameters.version)
        elif tag:
            subject = '%s v%s' % (self.sv_namecap, tag)
        else:
            subject = '%s v%s@%s' % (self.sv_namecap, branch, revision)

        return sdk2.task.ReleaseTemplate(
            ['skynet@yandex-team.ru'],
            subject,
            textwrap.dedent('''\
                %s

                Changelog:

                  ** Bugs
                     * SKYDEV-XXX: bug #1

                  ** Features
                     * SKYDEV-YYY: feature #1
            ''' % (subject, )),
            ['stable']
        )

    def arcadia_info(self):
        """ Return revision, tagname and branchname for release form. """
        parsed_url = sdk2.vcs.svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)
        path = parsed_url.path
        revision = parsed_url.revision
        if path.startswith('arc/branches/infra/%s' % (self.sv_name, )):
            branch = path.split('/')[4]
        else:
            branch = None

        if path.startswith('arc/tags/infra/%s' % (self.sv_name, )):
            tag = path.split('/')[4]
        else:
            tag = None

        return revision, tag, branch
