# -*- coding: utf-8 -*-

import textwrap

import requests
import lxml.etree

from sandbox import sdk2
from sandbox.sandboxsdk import parameters, svn, channel
from sandbox.common.errors import TaskError
from sandbox.projects.common.build import YaPackage
from sandbox.projects.skynet.abstract_build_skycore_bundle import SkycoreBundle  # compat


class BuildSkycoreYaPackage(YaPackage.YaPackage):
    """
        Build skycore yapackage.
        The result will be tgz bundle with skycore service descriptor inside.
    """
    platform = "linux_ubuntu_10.04_lucid"

    class ServiceVersion(parameters.SandboxStringParameter):
        name = 'service_version'
        description = 'Version to build'
        required = True

    input_parameters = [
        YaPackage.parameters.ArcadiaUrl,
        ServiceVersion,
        YaPackage.parameters.Musl,
        YaPackage.parameters.DoNotRemoveResources,
    ]

    execution_space = 4000

    resource_type = SkycoreBundle
    sv_name = 'my_service'
    sv_namecap = 'MyService'
    release_to = ['skynet@yandex-team.ru']

    @property
    def package_filename(self):
        return 'infra/%s/build.json' % (self.sv_name,)

    def on_enqueue(self):
        self.ctx[YaPackage.parameters.BuildType.name] = YaPackage.RELEASE
        self.ctx[YaPackage.PackageTypeParameter.name] = YaPackage.TARBALL
        self.ctx[YaPackage.PackagesParameter.name] = self.package_filename
        self.ctx[YaPackage.RunTestsParameter.name] = False
        self.ctx[YaPackage.ResourceTypeParameter.name] = self.resource_type.name
        self.ctx[YaPackage.PublishPackageParameter.name] = False

        self.ctx[YaPackage.StripBinariesParameter.name] = True
        self.ctx[YaPackage.CreateDebugPackagesParameter.name] = True
        self.ctx[YaPackage.CompressPackageArchiveParameter.name] = True
        self.ctx[YaPackage.UseNewFormatParameter.name] = True
        self.ctx[YaPackage.parameters.CheckoutParameter.name] = True

        version = self.ctx[self.ServiceVersion.name]
        self.ctx[YaPackage.CustomVersionParameter.name] = version
        self.ctx[YaPackage.PackageResourceDescriptionParameter.name] = {self.package_filename: "%s-%s" % (self.sv_name, version,)}

        YaPackage.YaPackage.on_enqueue(self)

    def mds_upload(self, path, store_path):
        base_uri = 'http://storage-int.mds.yandex.net:1111'
        get_uri_http = 'http://storage-int.mds.yandex.net/get-skynet'
        get_uri_https = 'https://storage-int.mds.yandex.net/get-skynet'

        auth = self.get_vault_data('SKYNET', 'skynet_mds_auth')

        headers = {
            'Authorization': 'Basic ' + auth,
        }

        with open(path, 'rb') as fp:
            response = requests.get(base_uri + '/upload-skynet/' + store_path, headers=headers, data=fp)

        if response.status_code == 200:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            uploaded_key = tree.get('key')

            if uploaded_key:
                return {
                    'http': get_uri_http + '/' + uploaded_key,
                    'https': get_uri_https + '/' + uploaded_key,
                }, uploaded_key

            return False
        elif response.status_code == 403:
            tree = lxml.etree.fromstring(response.text.encode('ascii'))
            key = tree.find('key').text

            if key:
                raise Exception('Unable to upload to mds: key already exists: %s' % (key, ))
            else:
                raise Exception('Unable to upload to mds: got 403 without key')
        elif response.status_code == 401:
            raise Exception('Unable to upload to mds: perm denied (code 401)')
        else:
            raise Exception('Unable to upload to mds: code %d' % (response.status_code, ))

    def on_release(self, additional_parameters):
        resources = channel.channel.rest.list_resources(task_id=self.id, resource_type=self.ctx[YaPackage.ResourceTypeParameter.name])
        if not resources or len(resources) > 1:
            raise TaskError("Exactly one SKYCORE_SERVICE resource expected")

        resource = resources[0]
        file_name = resource.file_name

        resource_path = self.sync_resource(resource.id)

        info = self.mds_upload(
            resource_path,
            '%s/%s' % (self.sv_name, file_name)
        )
        if not info:
            raise Exception('Unable to upload to MDS')

        mds_uri, mds_key = info

        channel.channel.sandbox.set_resource_attribute(resource.id, 'mds_key_service', 'skynet:' + mds_key)
        channel.channel.sandbox.set_resource_attribute(resource.id, 'version', self.ctx[self.ServiceVersion.name])
        channel.channel.sandbox.set_resource_attribute(resource.id, 'svn_url', self.ctx[YaPackage.parameters.ArcadiaUrl.name])

        for typ, uri in mds_uri.items():
            channel.channel.sandbox.set_resource_attribute(
                resource.id, '%s_mds' % (typ, ), uri
            )
            channel.channel.sandbox.set_resource_attribute(
                resource.id, '%s_mds_weight' % (typ, ), 50 if typ == 'http' else 51
            )

        for scheme in 'http', 'https':
            channel.channel.sandbox.set_resource_attribute(
                resource.id,
                scheme + '_sandbox_proxy',
                scheme + '://proxy.sandbox.yandex-team.ru/%d' % (resource.id, )
            )
            channel.channel.sandbox.set_resource_attribute(
                resource.id,
                scheme + '_sandbox_proxy_weight',
                10 if scheme == 'http' else 11
            )

        YaPackage.YaPackage.on_release(self, additional_parameters)

    @property
    def release_template(self):
        revision, tag, branch = self.arcadia_info()

        if self.ctx['service_version']:
            subject = '%s v%s' % (self.sv_namecap, self.ctx['service_version'],)
        elif tag:
            subject = '%s v%s' % (self.sv_namecap, tag,)
        else:
            subject = '%s v%s@%s' % (self.sv_namecap, branch, revision)

        return self.ReleaseTemplate(
            ['skynet@yandex-team.ru'],
            subject,
            textwrap.dedent('''\
                Changelog:
                  ** Bugs
                     * SKYDEV-XXX: bug #1
                  ** Features
                     * SKYDEV-YYY: feature #1
            '''),
            ['stable']
        )

    def arcadia_info(self):
        """ Return revision, tagname and branchname for release form. """
        parsed_url = svn.Arcadia.parse_url(self.ctx[YaPackage.parameters.ArcadiaUrl.name])
        path = parsed_url.path
        revision = parsed_url.revision
        if path.startswith('arc/branches/skynet'):
            branch = path.split('/')[3]
        else:
            branch = None

        if path.startswith('arc/tags/skynet'):
            tag = path.split('/')[3]
        else:
            tag = None

        return revision, tag, branch
