import logging
import os

from sandbox import sdk2

BACKUP_FINISHED_ATTR = "@backup_finished"
TIME_FORMAT = "%Y%m%d-%H%M%S"


class SmelterTransferBackupTask(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        src_cluster = sdk2.parameters.String("Source YT cluster", default="arnold")
        src_prefix = sdk2.parameters.String("Source YT prefix", default="//home/smelter/backup")
        dst_cluster = sdk2.parameters.String("Destination YT cluster", default="hahn")
        dst_prefix = sdk2.parameters.String("Destination YT cluster", default="//home/smelter/backup")
        vault_yt_token = sdk2.parameters.String("YT token vault item name", required=True)
        yt_pool = sdk2.parameters.String("YT pool")
        max_operations_count = sdk2.parameters.Integer("Max operations count in YT", default=5)
        transfer_manager_request_timeout = sdk2.parameters.Integer("TM request timeout", default=100000)
        transfer_manager_retry_count = sdk2.parameters.Integer("TM retry count", default=12)

    def make_tm_params(self):
        params = {}
        if self.Parameters.yt_pool:
            params["copy_spec"]["pool"] = self.Parameters.yt_pool
            params["postprocess_spec"]["pool"] = self.Parameters.yt_pool
        return params

    def make_yt_client(self, cluster=None):
        import yt.wrapper as yt
        yt_token = None
        if self.Parameters.value_yt_token:
            yt_token = sdk2.Vault.data(self.Parameters.vault_yt_token)
        return yt.YtClient(proxy=cluster, token=yt_token)

    def make_transfer_manager_client(self):
        from yt.transfer_manager.client import TransferManager

        return TransferManager(http_request_timeout=self.Parameters.transfer_manager_request_timeout, retry_count=self.Parameters.transfer_manager_retry_count)

    def on_execute(self):
        if self.Parameters.yt_pool:
            os.environ["YT_POOL"] = self.Parameters.yt_pool

        yt_src_client = self.make_yt_client(self.Parameters.src_cluster)
        yt_dst_client = self.make_yt_client(self.Parameters.dst_cluster)
        tm_client = self.make_transfer_manager_client()

        src_path = None
        for state in sorted(yt_src_client.list(self.Parameters.src_prefix), reverse=True):
            attr_finished = os.path.join(self.Parameters.src_prefix, state, BACKUP_FINISHED_ATTR)
            if yt_src_client.exists(attr_finished) and yt_src_client.get(attr_finished):
                src_path = os.path.join(self.Parameters.src_prefix, state)
                break

        if not src_path:
            raise Exception("Could not find last finished backup state")

        logging.info("Found last finished state %s", src_path)

        dst_path = os.path.join(self.Parameters.dst_prefix, state)
        dst_attr_finished = os.path.join(dst_path, BACKUP_FINISHED_ATTR)
        if yt_dst_client.exist(dst_attr_finished) and yt_dst_client.get(dst_attr_finished):
            logging.info("State %s is already transfered. Doing nothing", src_path)
            return

        src_dst_pairs = []
        for table in yt_src_client.list(src_path):
            logging.info("Transfering %s:%s/%s to %s:%s/%s",
                         self.Parameters.src_cluster, src_path, table,
                         self.Parameters.dst_cluster, dst_path, table)
            src_dst_pairs.append(tuple(os.path.join(src_path, table), os.path.join(dst_path, table)))

        tm_client.add_tasks_from_src_dst_pairs(
            src_dst_pairs,
            self.Parameters.src_cluster,
            self.Parameters.dst_cluster,
            params=self.make_tm_params(),
            sync=True,
            running_tasks_limit=self.Parameters.max_operations_count,
            enable_failed_tasks_restarting=True)

        yt_dst_client.set(dst_attr_finished, True)

        logging.info("Finished")
