# coding=utf-8
import json
import logging
import os
import time
import grp
import pwd

import requests
from sandbox import sdk2, common

from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.common import file_utils
from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess
from sandbox.sdk2.vcs.git import Git

logger = logging.getLogger(__name__)


LXC_CONTAINER_ID = 1046359220
SONIC_PLATFORM = 'mellanox'
SONIC_TARGET = 'sonic-mellanox.bin'
SONIC_REPO = 'https://github.com/Azure/sonic-buildimage.git'
DOCKER_REGISTRY = 'registry.yandex.net'
PLATFORM = 'mellanox'
TARGET = 'sonic-mellanox.bin'


class BuildSonicImage(sdk2.Task):
    '''Runs build of SONiC image
    '''

    class Requirements(sdk2.Requirements):
        privileged = True
        dns = DnsType.DNS64
        client_tags = Tag.IPV6

    class Parameters(sdk2.Task.Parameters):
        container = sdk2.parameters.Container(
            'Ubuntu trusty stable container with Docker',
            default_value=LXC_CONTAINER_ID,
            required=True,
        )

        with sdk2.parameters.Group("Sonic build settings") as sonic_block:
            sonic_platform = sdk2.parameters.Url(
                'PLATFORM',
                default=SONIC_PLATFORM,
                required=True
            )

            sonic_target = sdk2.parameters.Url(
                'SONIC build target',
                default=SONIC_TARGET,
                required=True
            )

        with sdk2.parameters.Group("Git repo") as git_block:
            sonic_repo = sdk2.parameters.Url(
                'SONiC repo url',
                default=SONIC_REPO,
                required=True
            )

            sonic_branch = sdk2.parameters.String(
                'Branch to build from',
                default='master',
                required=True
            )

            sonic_commit = sdk2.parameters.String("Commit to build from")

        with sdk2.parameters.Group("Git access Vault") as vault_block:
            ssh_vault_name = sdk2.parameters.String('Vault item with ssh key for git access')
            ssh_vault_owner = sdk2.parameters.String('Vault item owner')

        with sdk2.parameters.Group("Docker cache settings") as docker_block:
            docker_cache = sdk2.parameters.Bool('Use registry.yandex.net to cache build images')
            with docker_cache.value[True]:
                docker_login = sdk2.parameters.String('Docker login, default to task owner')
                docker_prefix = sdk2.parameters.String('Prefix to store images, registry.yandex.net/{prefix}/, default to docker login')
                oauth_vault_owner = sdk2.parameters.String('OAuth token vault owner, default to task owner')
                oauth_vault_key = sdk2.parameters.String('OAuth token vault key', default='docker')

    def on_prepare(self):
        # В on_prepare, а не on_execute потому что клонирование пишет в кеш Git-репозиториев, и это нельзя делать в
        # on_execute в тасках с privileged = True.
        self._clone_sonic_repository()

    def on_execute(self):
        self._build_params = ['SONIC_CONFIG_USE_NATIVE_DOCKERD_FOR_BUILD=y', 'USERNAME=admin', 'SONIC_ENABLE_RESTAPI=y']

        self._collect_debug_info_about_mount_points()
        self._configure_and_launch_docker_daemon()
        self._docker_login()
        self._install_extra_requirements()
        self._setup_user()

        self._extract_sonic_build_images()
        self._pull_docker_images()
        self._run_sonic_configure()
        self._push_docker_images()

        self._run_sonic_make_binary()

    def _clone_sonic_repository(self):
        with sdk2.ssh.Key(self, self.Parameters.ssh_vault_owner, self.Parameters.ssh_vault_name):
            git = Git(self.Parameters.sonic_repo)
            git.clone(self._sonic_repository_dir, branch=self.Parameters.sonic_branch)

            if self.Parameters.sonic_commit:
                git.checkout(self._sonic_repository_dir, commit=self.Parameters.sonic_commit)

            git.execute('submodule', 'update', '--init', '--recursive', cwd=self._sonic_repository_dir)

    def _configure_and_launch_docker_daemon(self):
        self._fix_ipv6()
        self._write_docker_config()
        self._restart_docker()

    def _fix_ipv6(self):
        self._collect_debug_info_about_network()
        try:
            # Костыли для IPv6 в Докер-контейнерах, взяты из скрипта таски BuildDockerImageV6
            # https://st.yandex-team.ru/MARKETINFRA-3566#5d307243a2b79e001c014ea3
            self._run(['ip6tables', '-t', 'nat', '-A', 'POSTROUTING', '-s', 'fd00::/8', '-j', 'MASQUERADE'])
            self._run(['sysctl', 'net.ipv6.conf.default.accept_ra=2'])
            self._run(['sysctl', 'net.ipv6.conf.all.forwarding=1'])
        finally:
            self._collect_debug_info_about_network()

    def _write_docker_config(self):
        self._write_json_to_log_and_file(
            '/etc/docker/daemon.json',
            {
                # Рекомендован здесь: https://docs.docker.com/storage/storagedriver/select-storage-driver/
                'storage-driver': 'overlay2',
                # Данные Докера должны быть в рабочей папке таски потому что вне рабочей папки OverlayFS.
                # Драйвер overlay2 не работает на OverlayFS.
                'data-root': self._docker_data_dir,
                # Включаем IPv6
                'ipv6': True,
                # Подсеть, из которой Докер будет выдавать IPv6-адреса контейнеров
                'fixed-cidr-v6': 'fd00::/8',
                # DNS-серверы, которые умеют отдавать IPv6-адреса для IPv4-only хостов
                'dns': ['2a02:6b8:0:3400::1023', '2a02:6b8:0:3400::5005'],
                # Говорим Докеру не настраивать сеть самостоятельно, у него не получается
                'iptables': False,
                'ip-forward': False,
                # Для совместимости с QYP
                'mtu': 1400,
                # For squashfs support
                'experimental': True,
            }
        )

    def _restart_docker(self):
        try:
            self._run(['systemctl', 'restart', 'docker.service'])
        finally:
            self._collect_debug_info_about_docker()

    def _docker_login(self):
	login = self.Parameters.docker_login
	if not login:
	    login = self.owner

        owner = self.Parameters.oauth_vault_owner
        if not owner:
            owner = self.owner

        docker_password = sdk2.Vault.data(owner, self.Parameters.oauth_vault_key)

        env = {'DOCKER_PASSWORD': docker_password}
        self._run(['docker login -u {} -p $DOCKER_PASSWORD {}'.format(login, DOCKER_REGISTRY)], env=env, shell=True)

    def _install_extra_requirements(self):
        self._run(['apt-get', 'update'])
        self._run(['apt-get', 'install', '-y', 'python-pip'])
        self._run(['pip', 'install', '--system', '--isolated', 'j2cli'])

    def _setup_user(self):
        try:
            pwd.getpwnam('docker')
        except:
            self._run(['useradd', '-g', 'docker', 'docker'])

        self._run(['chown', '-R', 'docker:docker', self._sonic_repository_dir])

    def _extract_sonic_build_images(self):
        showtag = self._run_with_stdout_captured(
            [
                'make',
                'showtag'
            ] + self._build_params,
            cwd=self._sonic_repository_dir,
            preexec_fn=self._preexec
        )

        self.docker_images = [line for line in showtag.split() if line.startswith('sonic-slave-')]

    def _pull_docker_images(self):
        prefix = self.Parameters.docker_prefix
        if not prefix:
            prefix = self.owner

        for image in self.docker_images:
            try:
                self._run(['docker', 'pull', '{}/{}/{}'.format(DOCKER_REGISTRY, prefix, image)])
                self._run(['docker', 'tag', '{}/{}/{}'.format(DOCKER_REGISTRY, prefix, image), image])
                self.set_info('Reusing docker image ' + image)
            except:
                pass

    def _push_docker_images(self):
        prefix = self.Parameters.docker_prefix
        if not prefix:
            prefix = self.owner

        for image in self.docker_images:
            self._run_ignoring_errors(['docker', 'tag', image, '{}/{}/{}'.format(DOCKER_REGISTRY, prefix, image)])
            self._run_ignoring_errors(['docker', 'push', '{}/{}/{}'.format(DOCKER_REGISTRY, prefix, image)])

    def _run_sonic_configure(self):
        self.set_info('starting SONiC make config')
        try:
            # comment out for debug info
            # self._run(['sed', '-i', '210i\\\\t@echo Running command $@', os.path.join(self._sonic_repository_dir, 'Makefile.work')])
            # self._run(['id'], preexec_fn=self._preexec)
            # self._run(['docker', 'info'], preexec_fn=self._preexec)

            self._run_with_custom_logger(
                'sonic-configure',
                [
                    'make',
                    '--trace',
                    'configure',
                    'PLATFORM={}'.format(self.Parameters.sonic_platform),
                ] + self._build_params,
                cwd=self._sonic_repository_dir,
                preexec_fn=self._preexec
            )

            # To dump SONiC configuration
            self.set_info(self._run_with_stdout_captured(
                ['make'] + self._build_params,
                cwd=self._sonic_repository_dir,
                preexec_fn=self._preexec
            ))
        except sdk2.helpers.ProcessLog.CalledProcessError as e:
            raise self.SonicbuildError(e)

    def _run_sonic_make_binary(self):
        self.set_info('starting SONiC make target/sonic-mellanox.bin')
        try:
            build_output_res = resource_types.BUILD_OUTPUT(
                self,
                'Build output',
                os.path.join(self._sonic_repository_dir, 'target'),
            )

            #build_output_res = resource_types.BUILD_OUTPUT(
            #    self,
            #    'sonic-mellanox.bin',
            #    os.path.join(self._sonic_repository_dir, 'target/sonic-mellanox.bin'),
            #)

            self._run_with_custom_logger(
                'sonic-make',
                [
                    'make',
                    '--trace',
                    'target/{}'.format(self.Parameters.sonic_target),
                ] + self._build_params,
                cwd=self._sonic_repository_dir,
                preexec_fn=self._preexec
            )
        except sdk2.helpers.ProcessLog.CalledProcessError as e:
            raise self.SonicbuildError(e)

    def _collect_debug_info_about_mount_points(self):
        self._run_ignoring_errors(['mount'])

    def _collect_debug_info_about_network(self):
        self._run_ignoring_errors(['ip', 'addr'])
        self._run_ignoring_errors(['ip', 'route'])

    def _collect_debug_info_about_docker(self):
        self._run_ignoring_errors(['systemctl', 'status', 'docker.service'])
        self._run_ignoring_errors(['journalctl', '-u', 'docker.service'])
        self._run_ignoring_errors(['docker', 'info'])

    def _run_ignoring_errors(self, args):
        try:
            self._run(args)
        except:
            pass

    def _run(self, args, **kwargs):
        # Лог запишется в common.log
        self._run_with_custom_logger(logging.getLogger(args[0]), args, **kwargs)

    def _run_with_custom_logger(self, logger, args, **kwargs):
        with sdk2.helpers.ProcessLog(self, logger=logger) as pl:
            # Так и задумано, stderr в stdout, чтобы всё сложилось в один файл, так удобнее.
            subprocess.check_call(args, stdout=pl.stdout, stderr=pl.stdout, **kwargs)

    def _run_with_stdout_captured(self, args, **kwargs):
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(args[0])) as pl:
            return subprocess.check_output(args, stderr=pl.stdout, **kwargs)

    @property
    def docker_login(self):
        login = self.Parameters.docker_login
        if not login:
            login = self.owner
        return login

    @property
    def _sonic_repository_dir(self):
        return str(self.path('sonic-buildimage'))

    @property
    def _docker_data_dir(self):
        return str(self.path('docker-data'))

    @staticmethod
    def _write_json_to_log_and_file(file, data):
        formatted_json = json.dumps(data, indent=4)
        logger.info('Writing to file %s: %s', formatted_json)
        file_utils.write_file(file, formatted_json)

    @staticmethod
    def _preexec():
        os.setgid(grp.getgrnam('docker').gr_gid)
        os.setuid(pwd.getpwnam('docker').pw_uid)
        os.environ['PYTHONPATH'] = '/usr/lib/python2.7/site-packages'
        if 'TERM' in os.environ:
            del os.environ['TERM']

    class SonicbuildError(common.errors.TaskFailure):
        def __init__(self, called_process_error):
            self.called_process_error = called_process_error

        def get_task_info(self):
            return (
                    '<br>'
                    'Build process failure. Please look at the logs'
                    '<br>'
                    + self.called_process_error.get_task_info()
            )
