# coding: utf-8

import os
import stat
import __builtin__

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.process import run_process

from sandbox.projects import resource_types
from sandbox.projects.common import constants as consts
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.build.parameters as build_params


def target2field_name(target):
    return 'build_%s' % target


class SortDCBinaries(parameters.SandboxInfoParameter):
    name = 'info_field'
    description = 'SortDC Binaries'


class CreateSymbolsArchive(parameters.SandboxBoolParameter):
    name = 'sortdc_create_symbols_archive'
    description = 'Create separate tarball with debug symbols'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class StripBinaries(parameters.SandboxBoolParameter):
    name = 'sortdc_strip_binaries'
    description = 'Strip result binaries'
    default_value = False
    group = build_params.BASE_BUILD_GROUP_NAME


class UseArcadiaApiFuse(build_params.UseArcadiaApiFuse):
    default_value = True


class YtStore(build_params.YtStore):
    default_value = True


def init(target_resources, custom_input_parameters):
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        CreateSymbolsArchive,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
        build_params.DefinitionFlags,
    ] + custom_input_parameters

    enabled_builds = {}
    for target_resource in target_resources:
        target = os.path.basename(target_resource.arcadia_build_path)
        field_name = target2field_name(target)
        build = __builtin__.type(
            field_name,
            (parameters.SandboxBoolParameter,),
            dict(name=field_name, description=target, default_value=True),
        )
        input_parameters.extend([build])
        enabled_builds[field_name] = target_resource

    return input_parameters, enabled_builds


class BaseBuildSortDCBins(YaMakeTask, nanny.ReleaseToNannyTask):
    type = 'BASE_BUILD_SORTDC_BINS'

    enabled_builds = {}
    input_parameters = [
        build_params.ArcadiaUrl,
        build_params.BuildSystem,
        build_params.BuildType,
        StripBinaries,
        UseArcadiaApiFuse,
        YtStore,
        build_params.YtProxy,
        build_params.YtDir,
        build_params.YtTokenVaultOwner,
        build_params.YtTokenVaultName,
        build_params.YtPut,
        build_params.YtStoreCodec,
        build_params.YtReplaceResult,
        build_params.Sanitize,
        build_params.ArcadiaPatch,
    ]
    TARGET_RESOURCES = ()

    def initCtx(self):
        YaMakeTask.initCtx(self)
        self.ctx['build_system'] = consts.YMAKE_BUILD_SYSTEM
        self.ctx[StripBinaries.name] = False
        # self.ctx[CreateSymbolsArchive.name] = True

    def get_targets(self):
        targets = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                targets.extend([os.path.dirname(resource_type.arcadia_build_path)])
        return targets

    def get_resources(self):
        resources = {}
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                resource = {
                    'description': resource_type.name,
                    'resource_type': resource_type,
                    'resource_path': os.path.basename(resource_type.arcadia_build_path),
                }
                resources[resource_type.name] = resource

        return resources

    def get_arts(self):
        arts = []
        for build_name, resource_type in self.enabled_builds.items():
            if self.ctx.get(build_name, False):
                arts.extend([{'path': resource_type.arcadia_build_path}])
        return arts

    def __copy_symbols(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--only-keep-debug', object_file, symbols_file],
            log_prefix='copy_symbols'
        )

    def __strip_binary(cls, object_file):
        mode = os.stat(object_file).st_mode
        os.chmod(object_file, mode | stat.S_IWUSR)
        process.run_process(
            ['strip', '-g', object_file],
            log_prefix='strip_binary'
        )
        os.chmod(object_file, mode)

    def __link_symbols_with_executable(cls, object_file, symbols_file):
        process.run_process(
            ['objcopy', '--add-gnu-debuglink', symbols_file, object_file],
            log_prefix='debuglink'
        )

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

        self.__process_symbols(pack_dir)

    def __process_symbols(self, pack_dir):
        create_symbols_archive = self.ctx[CreateSymbolsArchive.name]
        strip_symbols = self.ctx[StripBinaries.name]

        symbols_dir_name = 'symbols'
        symbols_dir = paths.make_folder(symbols_dir_name)

        for build_name, resource_type in self.enabled_builds.items():
            if not self.ctx.get(build_name):
                continue

            binary_name = os.path.basename(resource_type.arcadia_build_path)
            binary_path = os.path.join(pack_dir, binary_name)
            symbols_file = os.path.join(symbols_dir, binary_name + '.sym')

            if create_symbols_archive:
                self.__copy_symbols(binary_path, symbols_file)

            if strip_symbols or getattr(resource_type, 'always_strip', False):
                self.__strip_binary(binary_path)

            if create_symbols_archive:
                self.__link_symbols_with_executable(binary_path, symbols_file)

        if not create_symbols_archive:
            return

        symbols_archive_file = os.path.abspath('symbols.tar.gz')

        process.run_process(
            ['tar', 'czf', symbols_archive_file, symbols_dir_name],
            log_prefix='tar_symbols'
        )

        resource = self.create_resource(
            description='Build symbols',
            resource_path=symbols_archive_file,
            resource_type=resource_types.SORTDC_SYMBOLS_ARCHIVE,
            attributes={
                'ttl': 4
            }
        )

        self.mark_resource_ready(resource.id)

    def on_execute(self):
        self.ctx['ya_yt_token_vault_owner'] = 'UKROP-ROBOT'
        self.ctx['ya_yt_token_vault_name'] = 'sandbox-build-yt-token'
        YaMakeTask.on_execute(self)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        YaMakeTask.on_release(self, additional_parameters)

    # Use name option to rename dir
    def _create_svn_resource(self, path, resource_name, resource_type, source_dir):
        print(path, resource_name, resource_type)

        source_path = os.path.join(source_dir, path)
        source_dir = os.path.dirname(source_path)
        source_basename = os.path.basename(source_path)
        run_process(['tar', '-C', source_dir, '-zcf', resource_name, source_basename])
        self.create_resource(
            '',
            resource_name,
            resource_type)
