
# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build spellbook service bundle
    AUTHOR: @dvsonin
    GIT_REPO = https://bb.yandex-team.ru/scm/searchmon/spellbook.git
    ST_QUEUE: MARTY
"""
from __future__ import unicode_literals

import logging
import tarfile

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2 import helpers as sdk_helpers
from sandbox.sdk2.helpers import subprocess as sp

from sandbox.projects.common.nanny import nanny
from sandbox.projects.resource_types import PYTHON_BUNDLE2


class SpellbookBuildParameters(sdk2.Parameters):
    with sdk2.parameters.Group('Spellbook bundle build parameters') as wbjk_bundle:
        git_url = sdk2.parameters.String(
            'Git repository url',
            required=True
        )
        git_branch = sdk2.parameters.String(
            'Git branch or tag',
            default_value='master',
            required=True
        )
        requirements = sdk2.parameters.String('Requirements file name')
        use_xauth_token = sdk2.parameters.Bool('Use x-oauth-token in git url')
        oauth_val_name = sdk2.parameters.String('Vault secret name for access private repository')
        bit_bucket_username = sdk2.parameters.String('Bit Bucket username for authorization')


class BuildSpellbook(nanny.ReleaseToNannyTask2, sdk2.Task):
    name_for_humans = "build spellbook service bundle"

    Parameters = SpellbookBuildParameters

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    def on_execute(self):
        self.git()
        # self.build_bundle()
        self.create_resource()

    @staticmethod
    def run_process(*args, **kwargs):
        """
            Run subprocess with logging stdin and stdout
        :return: int
            Subprocess system exit code
        """
        logger_name = kwargs.pop('logger', 'system_shell')
        task = kwargs.pop('task', None)

        with sdk_helpers.ProcessLog(task, logger=logger_name) as _pl_:
            exit_code = sp.Popen(*args, stdout=_pl_.stdout, stderr=sp.STDOUT, **kwargs).wait()
            assert exit_code == 0
            return exit_code

    @staticmethod
    def checkout_gitrepo(git_url, branch, oauth=None, private=True, task=None, username=None, use_xtoken=True):
        """
            Clone and checkout passed git repository
        :param git_url: str
            Git repository url, for seems like this: https://bb.yandex-team.ru/scm/searchmon/spellbook.git
        :param branch: str
            Git branch or tag name
        :param oauth: str
            OAUTH creditals for access to private repository(bit bucket repos is private by default)
        :param private: bool
            Flag for mark repository is private
        :return: str
            Path to cloned repository
       """

        logging.info('-> check type of passed parameters')
        for p in [git_url, branch, oauth]:
            if not isinstance(p, (str, unicode)):
                raise TypeError(
                    'Invalid type of {}, a {} required for this value, {} taken'.format(
                        p,
                        str.__name__,
                        type(p).__name__
                    )
                )

        logging.info('-> checking out source from {}'.format(git_url))
        target_src = str(task.path('target_src'))

        if private:
            if oauth is not None:
                logging.info('->tockenaze git url with oauth tocken')
                if use_xtoken:
                    if git_url.startswith('https://'):
                        git_url = git_url.replace(
                            'https://',
                            'https://x-oauth-token:{}@'.format(oauth)
                        )
                    elif git_url.startswith('http://'):
                        git_url = git_url.replace(
                            'http://',
                            'http://x-oauth-token:{}@'.format(oauth)
                        )
                    else:
                        raise ValueError('Taken git url is not supported, git_url must be start with http or https')
                else:
                    if username is not None:
                        if git_url.startswith('https://'):
                            git_url = git_url.replace(
                                'https://',
                                'https://{u}:{t}@'.format(u=username, t=oauth)
                            )
                        elif git_url.startswith('http://'):
                            git_url = git_url.replace(
                                'http://',
                                'http://{u}:{t}@'.format(u=username, t=oauth)
                            )
                        else:
                            raise ValueError('Taken git url is not supported, git_url must be start with http or https')

            else:
                raise ValueError('Git repository is marked as private, but oauth token not taken')

        logging.info('-> Clone repository try git clone')
        BuildSpellbook.run_process(['git', 'clone', git_url, target_src], logger='git', task=task)

        if branch.startswith('refs/heads'):
            branch = branch[11:]

            logging.info('-> run git checkout from {}'.format(branch))
            BuildSpellbook.run_process(['git', 'checkout', branch], logger='git', task=task)

            logging.info('-> run git fetch')
            BuildSpellbook.run_process(['git', 'fetch'], logger='git', task='task')
        else:
            logging.info('->run git checkout {}'.format(branch))
            BuildSpellbook.run_process(['git', '--git-dir={}/.git'.format(target_src), 'checkout', branch], logger='git', task=task)

        return target_src

    def git(self):
        """
            Wrapper method for working with git
        """
        # Clone git repository
        if self.Parameters.oauth_val_name is not None:
            git_token = sdk2.Vault.data(self.owner, self.Parameters.oauth_val_name)
        else:
            git_token = None

        self.Context.target_src = BuildSpellbook.checkout_gitrepo(
            git_url=self.Parameters.git_url,
            branch=self.Parameters.git_branch,
            oauth=git_token,
            task=self,
            username=self.Parameters.bit_bucket_username,
            use_xtoken=self.Parameters.use_xauth_token
        )

    def create_resource(self, path=None, filename='spellbook_bundle.tar'):
        """
            Create tar archive with bundle and publish it
        """
        assert filename.endswith('.tar')

        _path_ = path or self.Context.resource_path
        _resource_ = str(self.path(filename))

        logging.info('-> compressing virtual environment')
        with tarfile.TarFile(_resource_, 'w') as _tar_:
            _tar_.add(_path_, 'ENV')

        _desc_ = '{} build from {}'.format(
            self.name_for_humans,
            self.Parameters.git_url
        )

        logging.info('-> Creating resource: {}'.format(filename))
        PYTHON_BUNDLE2(self, _desc_, _resource_)
