# -*- coding: utf-8 -*-
"""
    DESCRIPTION: Sandbox task for build spellbook sources bundle
    AUTHOR: @naumbi4
    REPO = https://a.yandex-team.ru/arc/trunk/arcadia/search/mon/spellbook
    ST_QUEUE: MARTY
"""

from __future__ import absolute_import, print_function, division

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.common.types import task as ctt

from sandbox.projects.common import constants
from sandbox.projects.common.build import YaPackage
from sandbox.projects.spellbook.resource_types import SpellbookPackage

SPELLBOOK_SOURCES_PATH = 'search/mon/spellbook/pkg.json'

USERNAME = 'robot-tickenator'

SUBTASK_YA_PACKAGE_TIMEOUT = 60 * 60 * 3
SUBTASK_TERMINAL_STATUSES = (
    ctt.Status.SUCCESS,
    ctt.Status.FAILURE,
    ctt.Status.EXCEPTION,
    ctt.Status.TIMEOUT,
    ctt.Status.STOPPED,
    ctt.Status.EXPIRED,
    ctt.Status.NO_RES,
    ctt.Status.DELETED,
)


class BuildSpellbookPackage(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        disk_space = 5 * 1024
        ram = 3 * 1024
        tasks_resource = None

    class Context(sdk2.Task.Context):
        ya_package_task_id = None
        users_hosts = None
        report_data = None

    class Parameters(sdk2.Task.Parameters):
        with sdk2.parameters.Group('Kernels build options') as kernels_build_options:
            checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl(
                'Svn url for arcadia',
                required=True,
                default_value=sdk2.svn.Arcadia.trunk_url(),
                description='For example, arcadia:/arc/trunk/arcadia@4975040'
            )

            spellbook_sources_path = sdk2.parameters.String(
                "Path to a kernels package.json",
                required=True,
                default=SPELLBOOK_SOURCES_PATH,
            )
        use_latest_sandbox_binary = sdk2.parameters.Bool(
            'Always use latest SANDBOX_TASKS_BINARY task resource',
            default=False,
            description='Makes sense for regular runs through Sandbox shchedulers'
        )

    def on_execute(self):
        with self.memoize_stage.stage_one:
            self._build_kernels()

        with self.memoize_stage.stage_two:
            self._prepare_kernels_resource()

    def _build_kernels(self):
        subtask = self._create_ya_package_task()
        subtask.enqueue()

        self.Context.ya_package_task_id = subtask.id

        raise sdk2.WaitTask(
            [subtask.id],
            SUBTASK_TERMINAL_STATUSES,
            wait_all=True,
            timeout=SUBTASK_YA_PACKAGE_TIMEOUT
        )

    def _create_ya_package_task(self):
        description = 'Build spellbook sources tarball'

        YaPackageClass = sdk2.Task['YA_PACKAGE']
        return YaPackageClass(
            self,
            # variables
            description=description,
            checkout_arcadia_from_url=self.Parameters.checkout_arcadia_from_url,
            packages=self.Parameters.spellbook_sources_path,
            # constants
            run_tests=False,
            use_aapi_fuse=True,
            aapi_fallback=True,
            use_new_format=True,
            package_type=YaPackage.TARBALL,
            build_type=constants.RELEASE_BUILD_TYPE,
            strip_binaries=True,
            resource_type=SpellbookPackage.name,
            save_build_output=True,
        )

    def _prepare_kernels_resource(self):
        expected_status = ctt.Status.SUCCESS
        task_id = self.Context.ya_package_task_id
        task = sdk2.Task[task_id]

        if task.status != expected_status:
            raise TaskFailure(
                'Subtask {} expected to be in {} status, but have {} status'.format(
                    task_id, expected_status, task.status
                )
            )

        resource = sdk2.Resource[SpellbookPackage.name] \
            .find(task=task) \
            .first()

        self._release_resource(resource)

    def _release_resource(self, resource):
        resource.ttl = 'inf'
        resource.released = 'stable'
