import logging
import os
import shutil
import tarfile
from contextlib import closing

from sandbox import sdk2
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.sandboxsdk import environments
from sandbox.projects.common.build.tasks import YaMakeTemplate
from sandbox.sdk2.helpers import subprocess
import sandbox.common.types.misc as ctm
from sandbox.projects.common.ya_deploy import release_integration
from sandbox.projects.sport.resource_types import (
    SPORT_NEW_BACKEND_CELERY_EXECUTABLE,
    SPORT_NEW_ADMIN_API_EXECUTABLE,
    SPORT_NEW_ADMIN_NGINX_CONFIG,
    SPORT_NEW_ADMIN_FRONT_BUILD,
    SPORT_NEW_BACKEND_ALEMBIC_EXECUTABLE,
)

npm_logger = logging.getLogger('npm_builder')


class BuildNewSportBackend(release_integration.ReleaseToYaDeployTask2, YaMakeTemplate.YaMakeTemplate):
    class Parameters(YaMakeTemplate.get_project_params([
        SPORT_NEW_BACKEND_CELERY_EXECUTABLE,
        SPORT_NEW_ADMIN_API_EXECUTABLE,
        SPORT_NEW_BACKEND_ALEMBIC_EXECUTABLE,
    ])):
        pass

    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64

        environments = [
            environments.NodeJS('10.20.1'),
            environments.GCCEnvironment(version="5.3.0"),
        ]

    YP_TOKEN_YAV_SECRET_ID = 'sec-01eh49y18pkvq6p2ryjwktpxt2'

    def on_execute(self):
        with arcadiasdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url) as arcadia_path:
            arcadia_path = sdk2.Path(arcadia_path)
            self.make_nginx_resource(arcadia_path)
            self.make_front_resource(arcadia_path)

        super(BuildNewSportBackend, self).on_execute()

    def make_front_resource(self, arcadia_path):
        resource = SPORT_NEW_ADMIN_FRONT_BUILD(self, "New Sport's admin front build", 'builds.tar')
        front_dir = str(arcadia_path / resource.arcadia_build_path)
        work_dir = 'front'
        shutil.copytree(front_dir, work_dir)

        with sdk2.helpers.ProcessLog(self, logger=npm_logger) as pl:
            subprocess.check_call(
                ['npm', 'install', '--production', '--registry=https://npm.yandex-team.ru'],
                cwd=work_dir,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )
            subprocess.check_call(
                ['npm', 'run', 'build'],
                cwd=work_dir,
                stdout=pl.stdout,
                stderr=pl.stderr,
            )

        build_path = os.path.join(work_dir, 'build')
        new_build_path = 'builds'
        shutil.move(build_path, new_build_path)

        resource_data = sdk2.ResourceData(resource)

        with closing(tarfile.open(str(resource_data.path), 'w')) as tar:
            tar.add(new_build_path)

        resource_data.ready()

    def make_nginx_resource(self, arcadia_path):
        resource = SPORT_NEW_ADMIN_NGINX_CONFIG(self, "New Sport's admin Nginx config", 'nginx_directory')
        resource_data = sdk2.ResourceData(resource)
        resource_data.path.mkdir(0o755, parents=True, exist_ok=True)

        configs = (
            'mime.types',
            'nginx.conf',
            'nginx.api.conf',
            'nginx.dashboard.conf',
            'nginx.cache.conf',
            'nginx.header.conf',
            'nginx.server.conf',
        )
        configs_path = arcadia_path / resource.arcadia_build_path

        for config in configs:
            shutil.copy(
                str(configs_path / config),
                str(resource_data.path / config)
            )
        resource_data.ready()
