import logging

from sandbox import sdk2, common
from sandbox.projects.sandbox.sandbox_lxc_image import SandboxLxcImage, UbuntuRelease
from sandbox.projects.sport.resource_types import SPORT_LXC_IMAGE
import sandbox.common.types.task as ctt


CUSTOM_SCRIPT = [
    'echo "custom_script start..." >> logs/stdout 2>> logs/stderr',
    'mkdir logs/',
    'mkdir /nvm/',
    'curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.34.0/install.sh | NVM_DIR=/nvm bash >> logs/stdout 2>> logs/stderr',
    'chmod +x /nvm/nvm.sh',
    '. /nvm/nvm.sh >> logs/stdout 2>> logs/stderr && nvm install 8.6.0 >> logs/stdout 2>> logs/stderr',
    'ln -s /nvm/versions/node/v8.6.0/bin/* /usr/bin/ >> logs/stdout 2>> logs/stderr',
    'mkdir -p /opt/nodejs/0.10/bin',
    'locale-gen ru_RU.UTF-8 >> logs/stdout 2>> logs/stderr',
    'echo "$PATH" >> logs/stdout 2>> logs/stderr',
    'echo "custom_script done." >> logs/stdout 2>> logs/stderr',
]

CUSTOM_PACKAGES = [
    'debhelper',
    'build-essential',
    'fuse',
    'cdbs',
    'git',
    'conductor-utils',
    'yandex-verstka-cdbs',
    'yandex-dh-environment',
    'yandex-dh-nodeubic',
    'yandex-dh-clearvcs',
]


class BuildSportLxcContainer(sdk2.Task):
    class Context(sdk2.Task.Context):
        build_task_id = 0

    class Requirements(sdk2.Task.Requirements):
        disk_space = 32 * 1024

    class Parameters(sdk2.Task.Parameters):
        description = "Build Sport LXC container"
        push_tasks_resource = True

        resource_description = sdk2.parameters.String(
            'Container resource''s custom description',
            default='Sport LXC container',
        )
        with sdk2.parameters.String(
            'Ubuntu release',
            default=UbuntuRelease.PRECISE,
        ) as ubuntu_release:
            # noinspection PyTypeChecker
            for release in UbuntuRelease:
                ubuntu_release.values[release] = release
        with sdk2.parameters.String(
            'Container resource type',
            required=True,
            default_value=SPORT_LXC_IMAGE.name,
        ) as resource_type:
            # noinspection PyTypeChecker
            for rt in sdk2.Resource:
                resource_type.values[rt.name] = rt.name
        custom_packages = sdk2.parameters.String(
            'List of packages to install during final stage, space-separated',
            multiline=True,
            default=' '.join(CUSTOM_PACKAGES),
        )
        base_packages = sdk2.parameters.String(
            'Space-separated list of packages to install in chrooted environment before building the container',
            multiline=True,
            default='yandex-ubuntu-archive-apt yandex-archive-keyring yandex-search-common-apt',
        )
        script = sdk2.parameters.String(
            'Shell script to execute during final stage',
            multiline=True,
            default='\n'.join(CUSTOM_SCRIPT),
        )

    def run_build_task(self):
        logging.info('Script: {}'.format(self.Parameters.script))

        lxc_params = SandboxLxcImage.Parameters

        params = {
            lxc_params.resource_type.name: self.Parameters.resource_type,
            lxc_params.resource_description.name: self.Parameters.resource_description,
            lxc_params.ubuntu_release.name: self.Parameters.ubuntu_release,
            lxc_params.test_result_lxc.name: True,
            lxc_params.set_default.name: False,
            lxc_params.custom_attrs.name: {},
            lxc_params.push_to_preprod.name: False,
            lxc_params.custom_image.name: True,
            lxc_params.install_common.name: False,
            lxc_params.custom_script.name: self.Parameters.script,
            lxc_params.base_packages.name: self.Parameters.base_packages,
            lxc_params.custom_packages.name: self.Parameters.custom_packages,
            lxc_params.unwanted_packages.name: 'juggler-client ^config-juggler',
        }

        build_task = SandboxLxcImage(
            self,
            description=self.Parameters.description,
            **params
        )

        self.Context.build_task_id = build_task.enqueue().id

        raise sdk2.WaitTask(
            self.Context.build_task_id,
            tuple(ctt.Status.Group.FINISH) + tuple(ctt.Status.Group.BREAK)
        )

    def on_execute(self):
        with self.memoize_stage.fire_once:
            self.run_build_task()

        build_task = sdk2.Task[self.Context.build_task_id]
        if build_task.status != ctt.Status.SUCCESS:
            raise common.errors.TaskFailure('Build image task failed, see child task for details')
