import os
import shutil

import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sdk2.helpers import subprocess
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.sport.resource_types import SPORT_NODE_API_SERVANT_PACKAGE, SPORT_LXC_IMAGE


class BuildSportNodeApiServant(nanny.ReleaseToNannyTask2, sdk2.Task):
    _working_path = 'build'

    class Parameters(sdk2.Parameters):
        max_restarts = 1
        kill_timeout = 1 * 3600

        with sdk2.parameters.Group('Config') as config_block:
            robot_name = sdk2.parameters.String(
                'Robot name',
                default='robot-sport-robot',
                required=True,
            )
            email = sdk2.parameters.String(
                'Email',
                default='sport-dev@yandex-team.ru',
                required=True,
            )

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64
        container_resource = sdk2.parameters.Container(
            resource_type=SPORT_LXC_IMAGE,
            default_value=3365321438,
            platform='linux_ubuntu_12.04_precise',
            required=True,
            owner='SPORT',
        )

    @property
    def _npm(self):
        """
        :return: path to `nmp` executable
        :rtype: str
        """
        return 'npm'

    @property
    def _node_modules_path(self):
        """
        :return: path to `node_modules`
        :rtype: str
        """
        return str(sdk2.Path('node_modules'))

    @property
    def _grunt(self):
        """
        :return: path to `grunt` executable
        :rtype: str
        """
        return str(sdk2.Path(self._node_modules_path).joinpath('grunt/bin/grunt'))

    def _setup_environ(self):
        os.environ["DEBFULLNAME"] = self.Parameters.robot_name
        os.environ["DEBEMAIL"] = self.Parameters.email
        os.environ["LC_ALL"] = 'ru_RU.UTF-8'
        os.environ["LANG"] = 'en_US.UTF8'
        os.environ['PATH'] = '/opt/nodejs/0.10/bin:' + os.environ['PATH']

    def on_execute(self):
        self._setup_environ()

        node_api_path = 'sport/node_api'
        arc_path = 'arcadia:/arc/trunk/arcadia/'

        with arcadiasdk.mount_arc_path(arc_path) as arcadia_path:
            arcadia_path = sdk2.Path(arcadia_path)
            node_api_dir = str(arcadia_path / node_api_path)
            shutil.copytree(node_api_dir, self._working_path)
        with sdk2.helpers.ProcessLog(self, logger="log.npm") as pl:
            subprocess.check_call(
                [self._npm, 'install', '--registry=https://npm.yandex-team.ru'],
                cwd=self._working_path,
                stderr=pl.stderr,
                stdout=pl.stdout,
            )
        with sdk2.helpers.ProcessLog(self, logger="log.grunt") as pl:
            subprocess.check_call(
                ['./' + self._grunt, 'build', '--nopublish=true', '--force'],
                cwd=self._working_path,
                stderr=pl.stderr,
                stdout=pl.stdout,
            )

        for f in [f for f in os.listdir(".") if os.path.isfile(f)]:
            if str(f).startswith("yandex-sport-api") and str(f).endswith(".deb"):
                file_name = str(self.path(f))
                shutil.move(f, file_name)
                SPORT_NODE_API_SERVANT_PACKAGE(self, "Sport Node Api Servant Package", file_name)
                break
