from __future__ import print_function

import base64
import io
import json
from logging import getLogger

import abc
import concurrent.futures
import requests
import six
import time
from typing import Iterable, List, BinaryIO, Tuple, Dict  # noqa

from sandbox import sdk2  # noqa

processing_log = getLogger('processing_log')


class DataFetcher(six.with_metaclass(abc.ABCMeta)):
    def __init__(self, task):
        """
        @type task: sdk2.Task
        """
        self._task = task

    @classmethod
    @abc.abstractmethod
    def get_type(cls):
        """
        @rtype: str
        """
        raise NotImplementedError('Method is not implemented [get_type]')

    @classmethod
    @abc.abstractmethod
    def get_project(cls):
        """
        @rtype: str
        """
        raise NotImplementedError('Method is not implemented [get_project]')

    @abc.abstractmethod
    def fetch(self):
        """
        @rtype: List[requests.Response]
        """
        raise NotImplementedError('Method is not implemented [fetch]')


# Infostrada
#

class InfostradaDataFetcher(six.with_metaclass(abc.ABCMeta, DataFetcher)):
    @classmethod
    def get_type(cls):
        """
        @rtype: str
        """
        return 'infostrada'

    def fetch(self):
        """
        @rtype: List[requests.Response]
        """
        responses = []
        for url in self.get_urls():
            response = requests.get(url, auth=self._get_auth_credentials())
            response.raise_for_status()
            responses.append(response)
        return responses

    @classmethod
    @abc.abstractmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        raise NotImplementedError

    def _get_auth_credentials(self):
        """
        @rtype: Tuple[str, str]
        """
        infostrada_secret = self._task.Parameters.infostrada_secret.data()
        return infostrada_secret['login'], infostrada_secret['password']


class BasketballDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'basketball'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        basketball_prefix = 'http://yandex.api.infostradasports.com/svc/TeamSports.svc/json/'
        basketball_keywords = {
            'languagecode': 'languageCode=2',
            'sportid': 'sportid=107',
            'eventphaseid': 'eventphaseid=140509',
            'teamid': 'teamid=681',
            'matchid': 'matchid=613999',
            'personid': 'personid=1368043',
            'phaseid': 'phaseid=220257',
        }
        urls_basketball = [
            'GetEditionList?{languagecode}&{sportid}',
            'GetPhaseList?{eventphaseid}&{languagecode}&{sportid}',
            'GetTeamList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetTable?{phaseid}&{languagecode}&{sportid}',
            'GetMatchList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetMatchLiveInfo?{languagecode}&{matchid}&{sportid}',
            'GetSquad?{eventphaseid}&{teamid}&{languagecode}&{sportid}',
            # 'GetPersonInfo?{personid}&{languagecode}&{sportid}',  # not bought
        ]
        return [
            basketball_prefix + url.format(**basketball_keywords)
            for url in urls_basketball
        ]


class FootballDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'football'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        football_prefix = "http://yandex.api.infostradasports.com/svc/Football.svc/json/"
        football_keywords = {
            'languagecode': 'languageCode=2',
            'editionid': 'editionid=22067',
            'phaseid': 'phaseid=118046',
            'teamid': 'teamid=1063',
            'matchid': 'matchid=2045275',
            'personid': 'personid=810584'
        }
        urls_football = [
            'GetEditionList?{languagecode}',
            'GetPhaseList?{editionid}&{languagecode}',
            'GetTeamList?{editionid}&{languagecode}',
            'GetTable?{phaseid}&{languagecode}',
            'GetMatchList_Edition?{editionid}&{languagecode}',
            'GetMatchInfo?{languagecode}&{matchid}',
            'GetMatchTeamStatList?{teamid}&{languagecode}&{matchid}',
            'GetSquad?{editionid}&{teamid}&{languagecode}',
            'GetPersonInfo?{personid}&{languagecode}',
            'GetMatchActionList_All?{matchid}&{languagecode}',
            'GetMatchLineup?{matchid}&{languagecode}',
            'GetTopScorerLiveList?{editionid}&{languagecode}',
            'GetMatchPersonStat?{matchid}&{personid}&{languagecode}',
            'GetEditionPersonStat?{editionid}&{personid}&{languagecode}',
        ]
        return [
            football_prefix + url.format(**football_keywords)
            for url in urls_football
        ]


class Formula1DataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'formula1'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        formula1_prefix = "http://yandex.api.infostradasports.com/svc/formula1.svc/json/"
        formula1_keywords = {
            'languagecode': 'languageCode=2',
            'eventphaseid': 'eventphaseid=1985260',
            'competitionid': 'competitionid=24114',
            'phaseid': 'phaseid=1985261',
            'teamid': 'TeamID=1500',
            'matchid': 'matchid=1875427',
            'season': 'season=2019',
            'rankingtype_driver': 'rankingtype=driver',
            'rankingtype_constructor': 'rankingtype=constructor',
        }
        urls_formula1 = [
            'GetGPList?{season}&{languagecode}',
            'GetGPParticipantList?{eventphaseid}&{languagecode}',
            'GetGPWeekend?{eventphaseid}&{languagecode}',
            'GetGPResult?{phaseid}&{languagecode}',
            'GetGPPhaseInfo?{phaseid}&{languagecode}',
            'GetGPResultLive?{phaseid}&{languagecode}',
            'GetGPParticipantList?{season}&{languagecode}&{eventphaseid}',
            'GetRanking?{season}&{rankingtype_driver}&{languagecode}',
            'GetRanking?{season}&{rankingtype_constructor}&{languagecode}',
        ]
        return [
            formula1_prefix + url.format(**formula1_keywords)
            for url in urls_formula1
        ]


class HockeyDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'hockey'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        hockey_prefix = 'http://yandex.api.infostradasports.com/svc/TeamSports.svc/json/'
        hockey_keywords = {
            'languagecode': 'languageCode=2',
            'sportid': 'sportid=113',
            'eventphaseid': 'eventphaseid=237520',
            'teamid': 'teamid=10554',
            'matchid': 'matchid=715277',
            'personid': 'personid=361479',
            'phaseid': 'phaseid=237521',
        }
        urls_hockey = [
            'GetEditionList?{languagecode}&{sportid}',
            'GetPhaseList?{eventphaseid}&{languagecode}&{sportid}',
            'GetTeamList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetTable?{phaseid}&{languagecode}&{sportid}',
            'GetMatchList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetMatchLiveInfo?{languagecode}&{matchid}&{sportid}',
            # 'GetMatchTeamStatList?{teamid}&{languagecode}&{matchid}&{sportid}',  # not bought
            'GetSquad?{eventphaseid}&{teamid}&{languagecode}&{sportid}',
            # 'GetPersonInfo?{personid}&{languagecode}&{sportid}', # not bought
            'GetMatchActionList_Goal?{matchid}&{languagecode}&{sportid}',
            'GetMatchLineup?{matchid}&{languagecode}&{sportid}',
        ]
        return [
            hockey_prefix + url.format(**hockey_keywords)
            for url in urls_hockey
        ]


class TennisDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'tennis'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        tennis_prefix = 'http://yandex.api.infostradasports.com/svc/Tennis.svc/json/'
        tennis_keywords = {
            'languagecode': 'languagecode=2',
            'season': 'season=2019',
            'eventphaseid': 'eventphaseid=1310925',
            'matchid': 'matchid=1178556',
            'personid': 'personid=797044',
        }
        urls_tennis = [
            'GetEditionList?{languagecode}&{season}',
            'GetPhaseList_EventPhase?{eventphaseid}&{languagecode}',
            'GetMatchList_EventPhase?{eventphaseid}&{languagecode}',
            'GetMatchLiveInfo?{matchid}&{languagecode}',
            'GetPersonInfo?{personid}&{languagecode}',
        ]
        return [
            tennis_prefix + url.format(**tennis_keywords)
            for url in urls_tennis
        ]


class TimeJudgeDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'time_judge'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        time_judge_prefix = "http://yandex.api.infostradasports.com/svc/TimeJudgeSports.svc/json/"
        time_judge_keywords = {
            'languagecode': 'languageCode=2',
            'editionid': 'editionid=2134048',
            'season': 'season=20182019',
            'sportid': 'sportid=218',
            'phaseid_for_result': 'phaseid=2182223',
            'phaseid_for_ranking': 'phaseid=2182223',
        }
        urls_time_judge = [
            'GetEditionList?{sportid}&{languagecode}',
            'GetPhaseList_Edition?{sportid}&{editionid}&{languagecode}',
            'GetResult?{sportid}&{phaseid_for_result}&{languagecode}',
            'GetRankingEditionList?{sportid}&{season}&{languagecode}',
            'GetRankingPhaseList_Edition?{sportid}&{season}&{editionid}&{languagecode}',
            'GetRanking?{phaseid_for_ranking}&sportid=218&{languagecode}',
        ]
        return [
            time_judge_prefix + url.format(**time_judge_keywords)
            for url in urls_time_judge
        ]


class VolleyballDataFetcher(InfostradaDataFetcher):
    @classmethod
    def get_project(cls):
        """
        @rtype: str
        """
        return 'volleyball'

    @classmethod
    def get_urls(cls):
        """
        @rtype: Iterable[str]
        """
        volleyball_prefix = 'http://yandex.api.infostradasports.com/svc/TeamSports.svc/json/'
        volleyball_keywords = {
            'languagecode': 'languageCode=2',
            'sportid': 'sportid=109',
            'eventphaseid': 'eventphaseid=213326',
            'teamid': 'teamid=710',
            'matchid': 'matchid=596087',
            'personid': 'personid=1368043',
            'phaseid': 'phaseid=1120795',
        }
        urls_volleyball = [
            'GetEditionList?{languagecode}&{sportid}',
            'GetPhaseList?{eventphaseid}&{languagecode}&{sportid}',
            'GetTeamList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetTable?{phaseid}&{languagecode}&{sportid}',
            'GetMatchList_EventPhase?{eventphaseid}&{languagecode}&{sportid}',
            'GetMatchLiveInfo?{languagecode}&{matchid}&{sportid}',
            'GetSquad?{eventphaseid}&{teamid}&{languagecode}&{sportid}',
            # 'GetPersonInfo?{personid}&{languagecode}&{sportid}',  # not bought
        ]
        return [
            volleyball_prefix + url.format(**volleyball_keywords)
            for url in urls_volleyball
        ]


# Main
#

FETCHERS = [
    BasketballDataFetcher,
    FootballDataFetcher,
    Formula1DataFetcher,
    HockeyDataFetcher,
    TennisDataFetcher,
    TimeJudgeDataFetcher,
    VolleyballDataFetcher,
]


def cache_data(task):
    """
    @type task: sdk2.Task
    @rtype: Dict[str, BinaryIO]
    """
    result_streams = {}
    common_stream = result_streams['common_cache'] = io.BytesIO()

    with concurrent.futures.ThreadPoolExecutor(max_workers=10) as executor:
        futures = {
            executor.submit(fetcher(task).fetch): {
                'type': fetcher.get_type(),
                'name': fetcher.__name__,
                'project': fetcher.get_project(),
            }
            for fetcher in FETCHERS
        }

        processing_log.info('Done 0/%s', len(futures))

        for idx, future in enumerate(concurrent.futures.as_completed(futures), start=1):
            fetcher = futures[future]
            stream = result_streams[fetcher['project']] = io.BytesIO()

            try:
                responses = future.result()
            except Exception as e:
                processing_log.error('Failed to fetch from %s: %s', fetcher['name'], repr(e))
                raise

            for response in responses:
                data = json.dumps({
                    'timestamp': time.time(),
                    'project': fetcher['project'],
                    'url': response.url,
                    'fetcher': {
                        'type': fetcher['type'],
                        'name': fetcher['name'],
                    },
                    'data': {
                        'content': base64.b64encode(response.content).decode(),
                        'status_code': response.status_code,
                        'headers': dict(response.headers)
                    }
                })
                print(data, file=stream)
                print(data, file=common_stream)

            processing_log.info('Done %s/%s [%s]', idx, len(futures), fetcher['name'])

    for stream in result_streams.values():
        stream.seek(0)

    return result_streams
