# -*- coding: utf-8 -*-
import json
import os
import os.path

from sandbox import sdk2
from sandbox.projects.resource_types import ARCADIA_PROJECT
from sandbox.projects.sport.resource_types import SPORT_PARTNERS_DATA_CACHE_DIFF

IMPORTER_DUMP_FILENAME = 'importer_dump.json'


class SportComparePartnersDataCacheTask(sdk2.Task):
    class Context(sdk2.Context):
        has_diff = False

    class Parameters(sdk2.Parameters):
        build1_resource = sdk2.parameters.Resource(
            label='Build 1', description='1st Tests build',
            resource_type=ARCADIA_PROJECT, required=True
        )
        build2_resource = sdk2.parameters.Resource(
            label='Build 2', description='2nd Tests build',
            resource_type=ARCADIA_PROJECT, required=True
        )

    def on_execute(self):
        build1_path = sdk2.ResourceData(self.Parameters.build1_resource).path
        build2_path = sdk2.ResourceData(self.Parameters.build2_resource).path

        dumps1 = _parse_build_output(build1_path)
        dumps2 = _parse_build_output(build2_path)

        out_resource = sdk2.ResourceData(SPORT_PARTNERS_DATA_CACHE_DIFF(
            task=self, description=self.Parameters.description, path='compare_result.txt'
        ))

        self.Context.has_diff = _save_diff(out_resource.path, dumps1, dumps2)

        out_resource.ready()


def _parse_build_output(path):
    dump_files = []
    for root, dirs, files in os.walk(str(path)):
        if IMPORTER_DUMP_FILENAME in files:
            test_path = root.split('/')
            project = test_path[test_path.index('test_importers') + 1]
            dump_files.append((project, os.path.join(root, IMPORTER_DUMP_FILENAME)))

    dumps = {}
    for project, filepath in dump_files:
        with open(filepath, 'r') as f:
            if project not in dumps:
                dumps[project] = {}
            dumps[project].update(json.load(f))

    return dumps


def _save_diff(path, dumps1, dumps2):
    missing = []
    differ = []

    for project, dumps1_value in dumps1.items():
        dumps2_value = dumps2.get(project)
        if not dumps2_value:
            missing.append({'key': project})
            continue
        for importer, value1 in dumps1_value.items():
            key = project + '.' + importer
            value2 = dumps2_value.get(importer)
            if not value2:
                missing.append({'key': key})
                continue
            if value1 != value2:
                differ.append({'key': key, 'old': value1, 'new': value2})
                continue

    diff_keys = []
    for collection in [missing, differ]:
        diff_keys.extend(it['key'] for it in collection)

    with path.open('w') as f:
        if not diff_keys:
            f.write(u'Ok\n')
            return []

        if missing:
            f.write(u'Missing:\n')
            for item in missing:
                f.write(u'\t- %s\n' % (item['key'], ))
            f.write(u'\n')

        if differ:
            f.write(u'Differ:\n')
            for item in differ:
                f.write(u'\t- %s\n' % (item['key'], ))
                f.write(u'\t\tOld: %s\n' % (json.dumps(item['old'], )))
                f.write(u'\t\tNew: %s\n' % (json.dumps(item['new'], )))
                f.write(u'\n')

    return diff_keys
