# coding=utf-8

import errno
import json
import re
import smtplib
import time
from datetime import datetime
from email.mime.text import MIMEText
from socket import error as SocketError

import requests

api_prefix = 'https://api.sport.yandex.ru/v2/'
admin_prefix = 'https://sport.news.yandex-team.ru/admin/'

translation_ru = 'ru'
translation_en = 'en'


def fetch(url):
    retries = 3
    retry_on = (403,)
    last_error = None
    for i in range(retries):
        try:
            response = requests.get(url, timeout=3)
            if response.status_code != 200:
                continue
            return json.loads(response.content)

        except (requests.Timeout, requests.ConnectionError) as e:
            last_error = e
        except requests.HTTPError as e:
            if e.code not in retry_on:
                raise
            last_error = e
        except SocketError as e:
            if e.errno != errno.ECONNRESET:
                raise
            last_error = e
    raise last_error


def download_all(url):
    skip = 0
    limit = 100

    url += '&' if '?' in url else '?'
    data = []
    while True:
        result = fetch(url + ('skip=%s&limit=%s' % (skip, limit)))
        data.extend(result)
        skip += limit
        if len(result) < limit:
            break

    return data


def now_timestamp():
    return int(time.time())


class Alert(object):
    def __init__(self, project, object_type, id, check_type, reason, model=None, model_id=None, field=None):
        self.project = project
        self.object_type = object_type
        self.id = id
        self.check_type = check_type
        self.reason = reason
        self.model = model
        self.model_id = model_id
        self.field = field

    def __eq__(self, other):
        return (self.reason == other.reason and self.model == other.model and self.model_id == other.model_id
                and self.field == other.field)

    def __hash__(self):
        return hash((self.reason, self.model, self.model_id, self.field))

    def get_object_sport_url(self):
        return 'https://yandex.ru/sport/event/%s' % self.id if self.id is not None else ''

    def get_object_api_url(self):
        if self.id is not None:
            return '%s%s/%ss/%s' % (api_prefix, self.project, self.object_type, self.id)
        return ''

    def get_object_admin_url(self):
        id = self.id[1:] if self.id.startswith('m') else self.id
        type = 'match' if self.object_type == 'event' else self.object_type
        return '%s%s/%s/%s/change/' % (admin_prefix, self.project, type, id)

    def get_model_admin_url(self):
        if self.field is None or self.model == 'Event':
            return self.get_object_admin_url()

        if self.check_type == 'translation':
            translation_key = '%s_%s_%s' % (self.model.lower(), self.model_id, self.field)
            return '%stranslations/trans/?q=%s' % (admin_prefix, translation_key)

        return ''


class Checker(object):
    _alerts = []

    def alert(self, check_type, reason, *args):
        raise NotImplementedError


class TranslationChecker(Checker):
    def check_translations(self):
        raise NotImplementedError

    def _check_translation_entry(self, obj, *args):
        if obj.get(translation_en) is None:
            self.alert('translation', 'empty_en_translation', *args)

        if obj.get(translation_ru) is None:
            self.alert('translation', 'empty_ru_translation', *args)
            return

        ru = obj[translation_ru]
        if not self.is_ru_str(ru):
            self.alert('translation', 'ru_translation_is_not_russian', *args)

    def is_ru_str(self, string):
        return not re.search(r'[a-zA-Z?]', string)


class Event(TranslationChecker):
    def __init__(self, project, event_data, now):
        self.project = project
        self._data = event_data
        self.now = now if now is not None else now_timestamp()
        self._alerts = []

    def alert(self, check_type, reason, *args):
        self._alerts.append(Alert(self.project, 'event', self._data.get('id'), check_type, reason, *args))

    def validate(self):
        self.check_status()
        self.check_start_time()
        self.check_translations()

        return self._alerts

    def check_status(self):
        # check: 'not_started' and 'in_progress'
        start_time = self._data.get('timestamp')
        if start_time is None:
            return

        status = self._data.get('status')
        if status is None:
            self.alert('status', 'empty_status', 'Event', self._data['id'])

        if status == 'not_started':
            if self.now - start_time > 5 * 60:  # 5 minutes
                self.alert('status', 'not_started', 'Event', self._data['id'])
        if status == 'in_progress':
            if self.now - start_time > 4 * 3600:  # 4 hours
                self.alert('status', 'in_progress', 'Event', self._data['id'])

    def check_start_time(self):
        timestamp = self._data.get('timestamp')
        if timestamp is None:
            return
        start_time_utc = datetime.utcfromtimestamp(timestamp)
        if start_time_utc.hour == 0 and start_time_utc.minute == 0:
            self.alert('start_time', 'suspicious_start_time', 'Event', self._data['id'])

    def check_translations(self):
        for field in ('name', 'stadium', 'city'):
            obj = self._data.get(field)
            if obj is None:
                continue
            self._check_translation_entry(obj, 'Event', self._data.get('id'), field)

        teams = self._data.get('teams', [])
        for team in teams:
            if team is None:
                continue
            for field in ('name', 'name_short'):
                obj = team.get(field)
                if obj is None:
                    continue
                self._check_translation_entry(obj, 'Team', team['id'], field)

        for action_type in ('goals', 'actions'):
            actions = self._data.get(action_type, [])
            for action in actions:
                for field, id_field in (('name', 'player_id'), ('subplayer_name', 'subplayer_id')):
                    obj = action.get(field)
                    if obj is None:
                        continue
                    self._check_translation_entry(obj, 'Person', self.get_person_id(action[id_field]), field)

        lineup = self._data.get('lineup')
        if lineup is not None:
            for entry in lineup:
                person = entry['person']
                for field in ('name', 'name_short'):
                    obj = person[field]
                    self._check_translation_entry(obj, 'Person', self.get_person_id(person['id']), field)

    def get_person_id(self, pid):
        if pid.startswith('p'):
            pid = pid[1:]
        return pid


class Competition(TranslationChecker):
    def __init__(self, project, id):
        self.project = project
        self.id = id
        self._data = fetch(api_prefix + project + '/competitions/' + id)
        self._alerts = []

    def alert(self, check_type, reason, *args):
        self._alerts.append(Alert(self.project, 'competition', self._data.get('id'), check_type, reason, *args))

    def validate(self):
        self.check_translations()

        return self._alerts

    def check_translations(self):
        for field in ['name', 'name_short']:
            obj = self._data.get(field)
            if obj is None:
                continue
            self._check_translation_entry(obj, 'Competition', self._data['id'], field)

        # TODO check phase's names and groups


def by_priority(x, y):
    return by_check_type(x, y) or by_project(x, y) or by_object(x, y)


def by_check_type(x, y):
    checks = {
        'status': 10,
        'start_time': 8,
        'translation': 6,
    }
    return checks.get(y.check_type, 0) - checks.get(x.check_type, 0)


def by_project(x, y):
    projects = [
        'football',
        'hockey',
        'basketball',
        'volleyball',
        'time_judge',
    ]
    projects_priority = {
        projects[i]: 10 - i for i in range(len(projects))
    }
    return projects_priority.get(y.project, 0) - projects_priority.get(x.project, 0)


def by_object(x, y):
    return int(y.id[1:]) - int(x.id[1:])  # 'm476155'


def make_report(alerts):
    message = ''

    last_check = None
    last_project = None
    last_id = None
    message += '<html><meta http-equiv="content-type" content="text/html; charset=utf-8"><body>'
    message += '<table width="100%" border="1" cellspacing="0" cellpadding="2">'
    for alert in sorted(alerts, cmp=by_priority):
        if alert.check_type != last_check:
            if last_check is not None:
                message += '</table>'
                message += '</td></tr></table>'
                message += '</td></tr></table>'
            message += '<tr><td>'
            message += 'Check: %s' % alert.check_type
            message += '<table width="100%" border="0">'
            last_check = alert.check_type
            last_project = None
            last_id = None
        if alert.project != last_project:
            if last_project is not None:
                message += '</table>'
                message += '</td></tr></table>'
            message += '<tr><td width="5%"></td><td>'
            message += 'Project: %s' % alert.project
            message += '<table width="100%" border="0">'
            last_project = alert.project
            last_id = None
        if alert.id != last_id:
            if last_id is not None:
                message += '</table>'
            message += '<tr><td width="5%"></td><td>'
            message += 'From %s with id=%s ' % (alert.object_type, alert.id)
            message += '(<a target="_blank" href=%s>sport</a>, <a target="_blank" href=%s>api</a>, <a target="_blank" href=%s>admin</a>)' % (
                alert.get_object_sport_url(), alert.get_object_api_url(), alert.get_object_admin_url()
            )
            message += '<table width="100%" border="0">'
            last_id = alert.id
        message += '<tr><td width="5%"></td><td>'
        message += '%s %s in %s with id=%s ' % (alert.reason, alert.field or '', alert.model, alert.model_id)
        message += '(<a target="_blank" href=%s>admin</a>)' % alert.get_model_admin_url()
        message += '</td></tr>'
    message += '</td></tr></table>'  # alert
    message += '</td></tr></table>'  # event
    message += '</td></tr></table>'  # project
    message += '</table>'  # check
    message += '</body></html>'

    return message


def alert(message, recipients):
    sender = 'robot-sport-robot@yandex-team.ru'
    msg = MIMEText(message, 'html')
    msg['Subject'] = 'Regular sport-api monitoring'
    msg['From'] = sender
    msg['To'] = ', '.join(recipients)

    s = smtplib.SMTP('omail.yandex.ru')
    s.sendmail(sender, recipients, msg.as_string())
    s.quit()


def check(projects, from_, to):
    alerts = set()
    for project in projects:
        competitions = set()
        url = api_prefix + project + '/events/?from=' + from_ + '&to=' + to
        now_ = now_timestamp()

        events = download_all(url)
        for event in events:
            if event['type'] == 'phase':
                continue
            competitions.add(event['competition_id'])
            event = Event(project, event, now_)
            event_alerts = event.validate()
            for a in event_alerts:
                alerts.add(a)

        for competition_id in competitions:
            competition = Competition(project, competition_id)
            competition_alerts = competition.validate()
            for a in competition_alerts:
                alerts.add(a)

    return alerts


def check_and_alert(projects, from_, to, recipients):
    alerts = check(projects, from_, to)
    message = make_report(alerts)
    alert(message, recipients)
