# -*- coding: utf-8 -*-

import gzip
import logging
import re
from collections import OrderedDict

from sandbox import sdk2

from sandbox.projects.sport.resource_types import SPORT_API_DIFF_LOGS
from . import api_client, log_parser
from .differ import Differ


class SportApiDiffByAccessLogs(sdk2.Task):
    """
    Сравнение ответов API по access-логам nginx-а
    """
    class Parameters(sdk2.Task.Parameters):
        canonical_base_url = sdk2.parameters.String("url prefix canonical source", required=True)
        tested_base_url = sdk2.parameters.String("url prefix tested source", required=True)
        archive_logs = sdk2.parameters.Resource("tar nginx access log", required=True)
        uri_pattern = sdk2.parameters.String("python regexp for uri filtering")
        sample_size = sdk2.parameters.Integer("sample size", default=1)
        common_headers = sdk2.parameters.Dict("common headers")
        canonical_headers = sdk2.parameters.Dict("canonical headers")
        tested_headers = sdk2.parameters.Dict("tested headers")
        retry_count = sdk2.parameters.Integer("count of retries API request", default=0)
        ignored_fields = sdk2.parameters.List("ignored fields (like root.some_key.some_subkey)", default=[])

    class Counter(object):
        def __init__(self):
            self.counters = OrderedDict()

        def count(self, it, key):
            self._init_counter(key)
            for item in it:
                self._add(key)
                yield item

        def add(self, key):
            self._init_counter(key)
            self._add(key)

        def _add(self, key):
            self.counters[key] += 1

        def _init_counter(self, key):
            if key not in self.counters:
                self.counters[key] = 0

    def on_execute(self):
        counter = self.Counter()
        data = sdk2.ResourceData(self.Parameters.archive_logs)
        common_headers = self.Parameters.common_headers or {}
        canonical_headers = self.Parameters.canonical_headers or {}
        tested_headers = self.Parameters.tested_headers or {}
        canonical_client = api_client.Client(
            self.Parameters.canonical_base_url,
            headers=self._merge_dicts(common_headers, canonical_headers),
            retry_cnt=self.Parameters.retry_count,
        )
        tested_client = api_client.Client(
            self.Parameters.tested_base_url,
            headers=self._merge_dicts(common_headers, tested_headers),
            retry_cnt=self.Parameters.retry_count,
        )
        with gzip.open(str(data.path)) as stream:
            i_request = log_parser.get_uri_from_parsed(log_parser.parse_log_line(stream))
            i_request = counter.count(i_request, "Total parsed logs")
            if self.Parameters.uri_pattern:
                i_request = counter.count(
                    _filter_uri(i_request, re.compile(self.Parameters.uri_pattern)), "Filtered by pattern \"{}\"".format(self.Parameters.uri_pattern))
            file_name = "diff_logs.log"
            i_dedup_uri = counter.count(_dedup_uri(i_request), "Deduped rows")
            i_sample_uri = counter.count(_sampled_uri(i_dedup_uri, self.Parameters.sample_size), "Sampled rows")
            with open(file_name, "w") as f:
                differ = Differ(self.Parameters.ignored_fields)
                for uri in i_sample_uri:
                    self._process_uri(canonical_client, tested_client, uri, f, counter, differ)
                    logging.log(logging.INFO, uri)
            SPORT_API_DIFF_LOGS(self, "Dump differ logs", file_name)

        self.set_info(self._get_info(counter.counters))

    def _process_uri(self, canonical_client, tested_client, uri, f, counter, differ):
        canonical_result = canonical_client.request(uri)
        tested_result = tested_client.request(uri)
        diffs = differ.get_diffs(canonical_result.data, tested_result.data)
        diffs = list(diffs)
        if diffs:
            counter.add("Total uri with diffs count")
            f.writelines(["---", uri, "\n"])
            for diff in diffs:  # type: differ.Diff
                f.writelines([str(diff), "\n\n"])
                logging.log(logging.INFO, diff.key, str(diff.canonical_val), str(diff.tested_val))
                counter.add("Total diffs count")
            f.write("\n")

    def _get_info(self, counters):
        return "\n".join("{}: {}".format(*i) for i in counters.items())

    def _merge_dicts(self, lhs, rhs):
        tmp = lhs.copy()
        tmp.update(rhs)
        return rhs


def _filter_uri(i_uri, pattern):
    for uri in i_uri:
        if pattern.match(uri):
            yield uri


def _sampled_uri(i_uri, sample_size):
    for i, uri in enumerate(i_uri):
        if i % sample_size:
            continue
        yield uri


def _dedup_uri(i_uri):
    seen_uri = set()
    for uri in i_uri:
        if uri in seen_uri:
            continue
        seen_uri.add(uri)
        yield uri
