class Differ(object):
    _none_object = object()

    def __init__(self, ignored_fields=None):
        self._ignored_fields = set(ignored_fields or [])

    def get_diffs(self, canonical, tested, key="root"):
        if key in self._ignored_fields:
            return

        if type(canonical) != type(tested):
            yield Diff(key, canonical, tested)
            return

        diffs = []
        if type(canonical) == dict:
            diffs = self._get_dict_diffs(canonical, tested, key)
        elif type(canonical) == list:
            diffs = self._get_list_diffs(canonical, tested, key)
        elif canonical != tested:
            yield Diff(key, canonical, tested)
            return

        for diff in diffs:
            yield diff

    def _get_dict_diffs(self, canonical, tested, parent):
        not_in_canonical = set(tested) - set(canonical)
        for key in not_in_canonical:
            for diff in self.get_diffs(self._none_object, tested[key], _get_key(parent, key)):
                yield diff
        for key, val in canonical.items():
            for diff in self.get_diffs(val, tested.get(key), _get_key(parent, key)):
                yield diff

    def _get_list_diffs(self, canonical, tested, parent):
        if len(canonical) != len(tested):
            yield Diff(parent, canonical, tested)
            return
        for i, val in enumerate(canonical):
            for diff in self.get_diffs(val, tested[i], _get_key(parent, "[{}]".format(i))):
                yield diff


def _get_key(parent, key):
    return "{}.{}".format(parent, key)


class Diff(object):
    def __init__(self, key, canonical_val, tested_val):
        self.key = key  # type: str
        self.canonical_val = canonical_val
        self.tested_val = tested_val

    def __str__(self):
        return "{}\n-{}\n+{}".format(self.key, self.canonical_val, self.tested_val)
