# -*- coding: utf-8 -*-

import os
import json

from sandbox import sdk2
from sandbox.common import errors
from sandbox.projects.resource_types import SPORT_WIZARD_DATA
from sandbox.projects.websearch.params import ResourceWithLastReleasedValueByDefault
from sandbox.common.types.task import Semaphores
from sandbox.sdk2.helpers import subprocess as subprocess
from sandbox.projects.websearch.upper import resources as upper_resources
import sandbox.sdk2.paths as paths
import sandbox.common.types.task as ctt
from sandbox.sandboxsdk import environments


class SportProxyDataBundle(sdk2.Resource):
    """
        Archive with the data for sport_proxy to deploy
    """
    version = sdk2.Attributes.Integer('Version', required=True)


class SportDataDeployConfig(sdk2.Resource):
    """
        Deploy config for sport_proxy.data
    """
    pass


class DeploySportProxyData(sdk2.Task):
    """
        Deploys the data for sport_proxy with bstr
    """

    class Requirements(sdk2.Task.Requirements):
        cores = 8
        environments = [
            environments.PipEnvironment('yandex-yt'),
        ]

    class Parameters(sdk2.Task.Parameters):
        build_and_release_data = sdk2.parameters.Bool("Build and release sport data", default=True)
        with build_and_release_data.value[True]:
            check_wizard_data = sdk2.parameters.Bool("Check wizard data before commit", default=True)
            yt_oauth_vault_name = sdk2.parameters.String("YT oauth key vault name", default='')
            tvm_secret_vault_name = sdk2.parameters.String("TVM secret vault name", default='')

        with build_and_release_data.value[False]:
            sport_data = ResourceWithLastReleasedValueByDefault(
                "Sport data to deploy",
                resource_type=SPORT_WIZARD_DATA,
                required=True,
            )

        deployer = ResourceWithLastReleasedValueByDefault(
            "search/tools/fast_data_deployment/deployer binary",
            resource_type=upper_resources.FastDataDeployer,
            required=True,
        )

        only_prepare = sdk2.parameters.Bool('Just prepare data, do not activate', default=False)
        deploy_config = sdk2.parameters.JSON('Deploy config', required=True)

        yt_token = sdk2.parameters.YavSecret(
            "Yav secret with YT deploy token",
            required=True,
        )
        yt_token_key = sdk2.parameters.String(
            "YT deploy token key name",
            required=True,
        )
        nanny_oauth = sdk2.parameters.YavSecret(
            "Yav secret with nanny deploy token",
            required=True,
        )
        nanny_oauth_key = sdk2.parameters.String(
            "Nanny deploy token key name",
            required=True,
        )

    def on_enqueue(self):
        super(DeploySportProxyData, self).on_enqueue()
        if not self.Requirements.semaphores:
            self.Requirements.semaphores = Semaphores(
                acquires=[
                    Semaphores.Acquire(name='lock-{}-deployer'.format(service), weight=2, capacity=2)
                    for service in self.Parameters.deploy_config["services"].keys()
                ],
            )

    def build_data_bundle(self, data_file):
        version_line = open(str(data_file), 'r').readline().strip().split(':', 1)
        if len(version_line) == 2 and version_line[0] == 'version':
            version = int(version_line[1])
        else:
            raise ValueError('No version info')
        archive_name = 'archive.tar.gz'
        dir_name = 'data_bundle'
        os.mkdir(dir_name)
        paths.copy_path(str(data_file), dir_name)
        with open(os.path.join(dir_name, '.svninfo'), 'w') as fd:
            fd.write('fast_data_version: %d\n' % version)
        archive_resource = SportProxyDataBundle(
            self, 'Archive with data for sport proxy', 'sport_proxy.tar.gz', version=version)
        archive_resource_data = sdk2.ResourceData(archive_resource)
        cmd = ["tar", "cvzf", str(archive_resource_data.path), dir_name]
        with sdk2.helpers.ProcessLog(self, logger="tar") as pl:
            subprocess.Popen(cmd, stdout=pl.stdout, stderr=subprocess.STDOUT).wait()
        archive_resource_data.ready()
        return archive_resource

    def save_deploy_config_resource(self, config_path):
        deploy_config_output_resource = SportDataDeployConfig(
            self, "Config used to deploy sport proxy data", config_path, ttl=90
        )
        with open(config_path, 'w') as config_fd:
            json.dump(self.Parameters.deploy_config, config_fd, indent=2)
        sdk2.ResourceData(deploy_config_output_resource).ready()
        return deploy_config_output_resource

    def get_sport_data_resource(self):
        if not self.Parameters.build_and_release_data:
            return self.Parameters.sport_data

        with self.memoize_stage.build_sport_data:
            self.set_info('Creating build sport_proxy.data task')
            task_class = sdk2.Task["BUILD_SPORT_PROXY_DATA"]
            sub_task = task_class(task_class.current,
                                  description="build last revision of sport_proxy.data from svn",
                                  owner=self.owner, priority=self.Parameters.priority,
                                  check_wizard_data=self.Parameters.check_wizard_data,
                                  yt_oauth_vault_name=self.Parameters.yt_oauth_vault_name,
                                  tvm_secret_vault_name=self.Parameters.tvm_secret_vault_name,
                                  restart_on_release=True)
            sub_task.Requirements.disk_space = 5000
            sub_task.save().enqueue()
            self.Context.build_subtask_id = sub_task.id
            raise sdk2.WaitTask([sub_task], ctt.Status.Group.FINISH | ctt.Status.Group.BREAK)

        with self.memoize_stage.release_sport_data:
            sub_task = sdk2.Task[self.Context.build_subtask_id]
            if sub_task.status not in ctt.Status.Group.SUCCEED:
                raise errors.TaskFailure('Build subtask failed')
            self.server.release(
                task_id=sub_task.id,
                type=ctt.ReleaseStatus.STABLE,
                subject="sport_proxy.data release",
                addresses_to="ustas@yandex-team.ru")
            raise sdk2.WaitTask([sub_task], ctt.Status.Group.RELEASE)

        with self.memoize_stage.release_check_sport_data:
            sub_task = sdk2.Task[self.Context.build_subtask_id]
            if sub_task.status == ctt.Status.NOT_RELEASED:
                raise errors.TaskFailure('Unable to release build task')

        sub_task = sdk2.Task[self.Context.build_subtask_id]
        resource = sdk2.Resource["SPORT_WIZARD_DATA"].find(task=sub_task).first()

        return resource

    def on_execute(self):
        sport_data_resource = self.get_sport_data_resource()
        deployer_path = str(sdk2.ResourceData(self.Parameters.deployer).path)
        config_path = str(sdk2.ResourceData(self.save_deploy_config_resource('deploy_config.json')).path)
        archive_resource = self.build_data_bundle(sdk2.ResourceData(sport_data_resource).path)
        archive_rbtorrent = archive_resource.skynet_id
        archive_version = archive_resource.version

        cmd = ['stdbuf', '-o', 'L', deployer_path, '--config', config_path, "--file", archive_rbtorrent,
               '--file-version', str(archive_version)]
        if self.Parameters.only_prepare:
            cmd.append('--only-prepare')

        deployer_env = os.environ.copy()
        deployer_env['YT_TOKEN'] = self.Parameters.yt_token.data()[self.Parameters.yt_token_key]
        deployer_env['OAUTH_NANNY'] = self.Parameters.nanny_oauth.data()[self.Parameters.nanny_oauth_key]

        self.set_info('Starting deployer:\n{}'.format(' '.join(cmd)))
        with sdk2.helpers.ProcessLog(self, logger="deployer") as pl:
            process = subprocess.Popen(
                cmd,
                env=deployer_env,
                stdin=subprocess.PIPE,
                stdout=subprocess.PIPE,
                stderr=pl.stderr,
                close_fds=True,
            )
            process.stdin.close()
            try:
                while True:
                    line = process.stdout.readline()
                    if not line:
                        break
                    self.set_info(line.strip('\n'))
                result = process.wait()
                self.set_info('Deployer finished')
                if result:
                    raise errors.TaskFailure('Failed to deploy')
            finally:
                process.stdout.close()
        self.set_info('Successfully deployed')
