
import logging
import kazoo
import os

from sandbox import sdk2
import sandbox.common.types.notification as ctn

from olymp_events import OlympEventsGenerator

logger = logging.getLogger(__name__)


class ZKClient(object):
    ZK_HOSTS = ('zookeeper1.search.yandex.net:2181,'
                'zookeeper2.search.yandex.net:2181,'
                'zookeeper3.search.yandex.net:2181,'
                'zookeeper4.search.yandex.net:2181,'
                'zookeeper5.search.yandex.net:2181')

    def __enter__(self):
        self.kz = kazoo.client.KazooClient(hosts=self.ZK_HOSTS, read_only=False)
        self.kz.start()
        return self.kz

    def __exit__(self, type, value, traceback):
        self.kz.stop()


class GenerateOlympEvents(sdk2.Task):
    """
        Generate olymp events periodically
    """

    class Parameters(sdk2.Task.Parameters):
        env = sdk2.parameters.String("olymp_events_env", default="", required=True, description='Environment for olymp_events script (dev, testing or prod)')

    ZK_LOCK_PATH = '/freshness/olymp_events_generate_lock'
    ZK_LOCK_TIMEOUT = 5

    def _execute(self):
        env = self.Parameters.env
        if env not in ['dev', 'testing', 'prod']:
            raise Exception('Bad env specified')

        generator = OlympEventsGenerator(env)
        generator.pushNewEvents()

        # Log file events_push.log is created by OlympEventsGenerator
        if os.path.exists('events_push.log'):
            logLines = open('events_push.log').read()
            if logLines:
                logging.info('Send email\n' + logLines)
                self.server.notification(
                    subject="Olymp pushes sent",
                    body=logLines,
                    recipients=["iv-ivan@yandex-team.ru", "ustas@yandex-team.ru"],
                    transport=ctn.Transport.EMAIL
                )
            else:
                logging.info('No data in events_push.log')

    def on_execute(self):
        # Only one instance of this task should be executed
        with ZKClient() as zk:
            lockPath = self.ZK_LOCK_PATH
            lock = zk.Lock(lockPath)
            try:
                if not lock.acquire(blocking=True, timeout=self.ZK_LOCK_TIMEOUT):
                    logging.info("Lock not aquired. Exit")
                    return
            except kazoo.exceptions.LockTimeout:
                logging.info("Lock not aquired. Exit")
                return

            try:
                return self._execute()
            finally:
                lock.release()
