#!/usr/bin/env python
# -*- coding: utf-8 -*-

import json
import time
import urllib2


def robustFetch(url, retries=3):
    lastEx = None
    for numTry in range(retries):
        try:
            resp = urllib2.urlopen(url).read()
            return resp
        except urllib2.HTTPError as e:
            if e.code == 404:
                # Don't throw exception for 404 code (sometimes match is returned at CATALOG_URL, but does not exist at MATCH_URL)
                return None
            else:
                lastEx = e
                time.sleep(1.0)
    raise Exception("Failed to open url %s: %s" % (url, str(lastEx)))


class OlympCatalog:
    CATALOG_URL = 'https://proxy.sandbox.yandex-team.ru/last/SPORT_WIZARD_DATA'
    EVENT_URL = 'http://api.sport.yandex.ru/v2/olympic/20172018/events/{eventId}'
    COUNTRY_MEDALS_URL = 'https://api.sport.yandex.ru/v2/olympic/20172018/countries/{country}/medals-list'
    PERSONS_URL = 'http://api.sport.yandex.ru/v2/olympic/20172018/persons/{person_id}'

    olymp_timetable_item_prefix = "olymp_timetable_item:"
    olymp_event_prefix = "olymp_event:"

    def __init__(self):
        pass

    def _getCatalogData(self):
        resp = robustFetch(self.CATALOG_URL)

        olympTimetableItems = {}
        olympEvents = {}
        for line in resp.split("\n"):
            if line.startswith(self.olymp_timetable_item_prefix):
                itemInfo = json.loads(line[len(self.olymp_timetable_item_prefix):])
                olympTimetableItems[itemInfo["id"]] = itemInfo
            if line.startswith(self.olymp_event_prefix):
                eventInfo = json.loads(line[len(self.olymp_event_prefix):])
                olympEvents[eventInfo["id"]] = eventInfo
        ids = set(olympTimetableItems.keys()) & set(olympEvents.keys())

        # merge event info in one dict
        olympEventsFullInfo = {}
        for id in ids:
            olympEventsFullInfo[id] = olympTimetableItems[id]
            olympEventsFullInfo[id].update(olympEvents[id])
        return olympEventsFullInfo

    def getEventsForPeriod(self, tsFrom, tsTo):
        catalogData = self._getCatalogData()
        for eventId, eventInfo in catalogData.iteritems():
            tsStart = eventInfo['ts_start']
            if ((tsStart is not None) and (tsStart >= tsFrom) and (tsStart <= tsTo)):
                yield eventInfo

    def getEventInfo(self, eventId):
        eventUrl = self.EVENT_URL.format(eventId=eventId)
        resp = robustFetch(eventUrl)
        # Response may be empty (sometimes match is returned via CATALOG_URL, but not exists via EVENT_URL)
        return json.loads(resp) if resp else None

    def getCountryMedals(self, country):
        medalsUrl = self.COUNTRY_MEDALS_URL.format(country=country)
        resp = robustFetch(medalsUrl)
        return json.loads(resp) if resp else None

    def getPersonInfo(self, personId):
        if not personId:
            return None
        personsUrl = self.PERSONS_URL.format(person_id=personId)
        resp = robustFetch(personsUrl)
        return json.loads(resp)
