#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import json
import re
import requests
import sys
import time
import urllib

from olymp_catalog import OlympCatalog


###
# Event - some competition on olympiad
# EventUpdate - status update of some event
###


CONFIG = {
    'dev': {
        'mongoHost': 'localhost',  # behemoth.search.yandex.net
        'mongoPort': 27017,
        'databaseName': 'olymp_events',
        'eventsPushLog': 'events_push.log',
        'sendPushUrl': 'http://push-beta.n.yandex-team.ru/pushes',
    },
    'testing': {
        'mongoReplicaSet': 'mongodb://man1-7436.search.yandex.net:22560,sas1-7558.search.yandex.net:29360,vla1-4211.search.yandex.net:22560/?replicaSet=sportpush',
        # 'mongoReplicaSet' : 'mongodb://man1-6606.search.yandex.net:14200,sas1-6967.search.yandex.net:9680,ws31-418.search.yandex.net:14120/?replicaSet=test_sportpush',
        'mongoWriteReplicas': 2,    # write at least to 2 replicas (master and 1 secondary)
        'mongoWaitJournaling': True,  # during write wait until journaling finished
        'databaseName': 'olymp_events_testing',
        'eventsPushLog': 'events_push.log',
        'sendPushUrl': 'http://push-beta.n.yandex-team.ru/pushes',
    },
    'prod': {
        'mongoReplicaSet': 'mongodb://man1-7436.search.yandex.net:22560,sas1-7558.search.yandex.net:29360,vla1-4211.search.yandex.net:22560/?replicaSet=sportpush',
        # 'mongoReplicaSet' : 'mongodb://man1-6606.search.yandex.net:14200,sas1-6967.search.yandex.net:9680,ws31-418.search.yandex.net:14120/?replicaSet=test_sportpush',
        'mongoWriteReplicas': 2,
        'mongoWaitJournaling': True,
        'databaseName': 'olymp_events',
        'eventsPushLog': 'events_push.log',
        'sendPushUrl': 'http://sup.yandex.net/pushes',
    },
}


ONE_MINUTE = 60
HALF_HOUR = 30 * 60
ONE_HOUR = 60 * 60
FIVE_HOURS = 5 * ONE_HOUR
TEN_HOURS = 10 * ONE_HOUR
ONE_DAY = 24 * ONE_HOUR


CODE_TO_SPORT = {
    'CUR': 'curling',
    'IH': 'hockey',
    'IHO': 'hockey'
}


def get_sport(sport_code):
    return CODE_TO_SPORT.get(sport_code, 'other')


SPORT_EVENTS = {
    'curling': [
        'need_start_alert',
        'in_progress',
        'finished'
    ],
    'hockey': [
        'need_start_alert',
        'in_progress',
        # 'shootout_5',
        'goal',
        'finished'
    ],
    'other': [
        'need_start_alert',
        'in_progress',
        'finished'
    ],
}

# gender must be ", ---" or ""
# stage must be ", ---" or ""
# Stage must be ". ^--" or ""
EVENT_MESSAGES = {
    'ru': {
        'short': {
            'curling_need_start_alert': {'default': [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начнётся в {time}']},
            'curling_in_progress': {'default':
                                             [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начался — смотрите трансляцию'],
                                          'no_translation':
                                             [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начался']
                                        },
            'curling_finished': {'default': [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч окончен со счетом {score}']},

            'hockey_need_start_alert': {'default': [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начнётся в {time}']},
            'hockey_in_progress': {'default':
                                            [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начался — смотрите трансляцию'],
                                         'no_translation':
                                            [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начался']
                                        },
            'hockey_shootout_5': {'default': [u'{team1} — {team2}', u'Началась серия буллитов']},
            'hockey_goal': {'default': [u'Гол!', u'{team1} — {team2}: {score}']},
            'hockey_finished': {'default': [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч окончен со счетом {score}']},

            'other_need_start_alert': {'default': [u'{Sport}{gender}', u'В {time} — {details}{stage}']},
            'other_in_progress': {'default':
                                            [u'{Sport}{gender} — трансляция', u'Началось соревнование: {details}{stage}'],
                                        'no_translation':
                                            [u'{Sport}{gender}', u'Началось соревнование: {details}{stage}']
                                        },
            'other_finished': {'default': [u'{Sport}{gender}', u'Итоги: {details}{stage}']},
        },
        'long': {
            'curling_need_start_alert': {'default': [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начнётся в {time}']},
            'curling_in_progress': {'default':
                                             [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начался — смотрите трансляцию'],
                                          'no_translation':
                                             [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч начался']
                                        },
            'curling_finished': {'default': [u'Кёрлинг{gender}', u'{team1} — {team2}. Матч окончен со счетом {score}']},

            'hockey_need_start_alert': {'default': [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начнётся в {time}']},
            'hockey_in_progress': {'default':
                                            [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начался — смотрите трансляцию'],
                                         'no_translation':
                                            [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч начался']
                                        },
            'hockey_shootout_5': {'default': [u'{team1} — {team2}', u'Началась серия буллитов']},
            'hockey_goal': {'default': [u'Гол!', u'{team1} — {team2}: {score}']},
            'hockey_finished': {'default': [u'Хоккей{gender}{Stage}', u'{team1} — {team2}. Матч окончен со счетом {score}']},
            # below VVV statements which differe from short verison
            'other_need_start_alert': {'default': [u'{Sport}', u'{Details}{gender}{Stage}. Начало в {time}']},
            'other_in_progress': {'default':
                                            [u'{Sport} — трансляция', u'{Details}{gender}{Stage}. Соревнование началось'],
                                        'no_translation':
                                            [u'{Sport}', u'{Details}{gender}{Stage}. Соревнование началось']
                                        },
            'other_finished': {'default': [u'{Sport}', u'{Details}{gender}{Stage}. Итоги']},
        }
    }
}

# Name must be "^--. " or ""
# gender must be ", ---" or ""
MEDAL_MESSAGE = [u'{Medal} у российских спортсменов', u'{Name}{Sport}{gender}. {Details}']

MEDAL_NAMES = {
    'gold': u"золото",
    'silver': u"серебро",
    'bronze': u"бронза",
}

MONTH_NAMES_GENITIVE = {
    'ru': {
        1: u'января',
        2: u'февраля',
        3: u'марта',
        4: u'апреля',
        5: u'мая',
        6: u'июня',
        7: u'июля',
        8: u'августа',
        9: u'сентября',
        10: u'октября',
        11: u'ноября',
        12: u'декабря'
    }
}

COMMON_STRINGS = {
    'ru': {
        'by_shootout': u'по буллитам',
        'msk_timezone': u'мск',
        'video': u'видео',
        'men': u'мужчины',
        'women': u'женщины',
        'open': u'',
        'mixed': u'',
    }
}


def capitalize_first_letter(string):
    return string[0].upper() + string[1:]


def uncapitalize_first_letter(string):
    return string[0].lower() + string[1:]


class OlympEventsDB:
    def __init__(self, cfgName='dev'):
        from pymongo import MongoClient  # , ReturnDocument

        config = CONFIG[cfgName]
        kwOptions = {}
        if 'mongoWriteReplicas' in config:
            kwOptions['w'] = config['mongoWriteReplicas']
        if 'mongoWaitJournaling' in config:
            kwOptions['j'] = config['mongoWaitJournaling']
        if 'mongoReplicaSet' in config:
            self.mongoClient = MongoClient(
                config['mongoReplicaSet'],
                **kwOptions
            )
        else:
            self.mongoClient = MongoClient(
                host=config['mongoHost'],
                port=config['mongoPort'],
                **kwOptions
            )
        self.db = self.mongoClient[config['databaseName']]

    def _getNextId(self, counterName):
        '''
        # This was used in pymongo 3.2.1, in sandbox the older version is used
        ret = self.db.counters.find_one_and_update(
            {'_id': counterName},
            {'$inc': {'seq': 1}},
            upsert=True,
            return_document=ReturnDocument.AFTER
        )
        '''
        ret = self.db.counters.find_and_modify(
            {'_id': counterName},
            {'$inc': {'seq': 1}},
            upsert=True,
            new=True
        )
        return ret['seq']

    def getEventState(self, eventId, generateDefault=True):
        event = self.db.event_states.find_one({'event_id': eventId})
        if event:
            return event
        else:
            if generateDefault:
                # default state
                return {
                    'status': 'not_started',
                    'result': [0, 0],
                }
            else:
                return None

    def setEventState(self, eventId, state):
        self.db.event_states.update(
            {'event_id': eventId},
            {'$set': state},
            upsert=True
        )

    def addNewEventUpdates(self, eventUpdates):
        from pymongo.errors import DuplicateKeyError

        for eventUpdate in eventUpdates:
            try:
                eventUpdate.update({
                    '_id': self._getNextId('eventUpdate_id')
                })
                self.db.event_updates.insert(eventUpdate)
            except DuplicateKeyError:
                print >>sys.stderr, "Try to insert duplicate event: event_id=%s, update_key=%s" % (eventUpdate['event_id'], eventUpdate['update_key'])

    def addNewMedals(self, newMedals):
        from pymongo.errors import DuplicateKeyError

        for medal in newMedals:
            try:
                medal.update({
                    '_id': self._getNextId('medal_id')
                })
                self.db.medals.insert(medal)
            except DuplicateKeyError:
                print >>sys.stderr, "Try to insert duplicate medal: id=%s" % (medal['id'])

    def getLastEventUpdates(self, fromId=0, fromTs=None):
        eventUpdates = []
        lastId = fromId
        findCondition = {'_id': {'$gt': fromId}}
        if fromTs:
            findCondition['add_time'] = {'$gt': fromTs}
        for eventUpdate in self.db.event_updates.find(findCondition):
            eventUpdates.append(eventUpdate)
            lastId = max(lastId, eventUpdate['_id'])
        return (eventUpdates, lastId)

    def getLastMedals(self, fromId=0, fromTs=None):
        medals = []
        lastId = fromId
        findCondition = {'_id': {'$gt': fromId}}
        if fromTs:
            findCondition['add_time'] = {'$gt': fromTs}
        for medal in self.db.medals.find(findCondition):
            medals.append(medal)
            lastId = max(lastId, medal['_id'])
        return (medals, lastId)

    def logOlympResponse(self, respKey, respData):
        self.db.olymp_response_log.insert_one({
            'time': time.time(),
            'key': respKey,
            'data': respData
        })

    def clearOldOlympResponses(self, maxAgeDays=14):
        self.db.olymp_response_log.delete_many(
            {'time': {'$lt': time.time() - maxAgeDays * ONE_DAY}}
        )

    def getClusterVar(self, name):
        var = self.db.cluster_vars.find_one({'name': name})
        return var['value'] if var else None

    def setClusterVar(self, name, value):
        self.db.cluster_vars.update(
            {'name': name},
            {'$set': {'name': name, 'value': value}},
            upsert=True
        )

    def getMedalIds(self, country):
        medal_ids = []
        for medal in self.db.medal_ids.find({'name': 'medal_ids'}):
            medal_ids.append(medal["id"])
        return medal_ids

    def updateMedalIds(self, country, newMedalIds):
        from pymongo.errors import DuplicateKeyError

        for id in newMedalIds:
            try:
                self.db.medal_ids.insert({'name': 'medal_ids', 'id': id})
            except DuplicateKeyError:
                pass

    def getSecurityVar(self, name):
        var = self.db.security_vars.find_one({'name': name})
        return var['value'] if var else None

    def logSendPush(self, eventId, msg):
        self.db.send_push_log.insert({
            'time': datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S'),
            'event_id': eventId,
            'msg': msg
        })


class OlympEventsGenerator:
    def __init__(self, configName, eventsMaxAge=ONE_HOUR, checkEventsAge=TEN_HOURS, startAlertTime=HALF_HOUR):
        self.eventsMaxAge = eventsMaxAge
        self.checkEventsAge = checkEventsAge
        self.startAlertTime = startAlertTime
        self.config = CONFIG[configName]
        self.logOlympResponses = True

        self.db = OlympEventsDB(configName)
        self.lang = 'ru'
        self.sportTypes = ['curling', 'hockey', 'other']

        self.eventUpdateTypeToOrder = {}
        for sport in self.sportTypes:
            self.eventUpdateTypeToOrder[sport] = {}
            n = 0
            for ev in SPORT_EVENTS[sport]:
                self.eventUpdateTypeToOrder[sport][ev] = n
                n += 1

    def _updateKeyToType(self, updateTypeKey):
        return 'goal' if updateTypeKey[:6] == 'score_' else updateTypeKey

    def _getUpdateOrder(self, eventUpdate):
        return self.eventUpdateTypeToOrder[eventUpdate['sport']].get(self._updateKeyToType(eventUpdate['update_key']), -1)

    def _leaveLastUpdateForEachEvent(self, eventsUpdates):
        updatesByEvent = {}
        for eventUpdate in eventsUpdates:
            if eventUpdate['event_id'] not in updatesByEvent:
                updatesByEvent[eventUpdate['event_id']] = eventUpdate
            else:
                curEventUpdate = updatesByEvent[eventUpdate['event_id']]
                curEventUpdateOrder = self._getUpdateOrder(curEventUpdate)
                newEventUpdateOrder = self._getUpdateOrder(eventUpdate)
                if (newEventUpdateOrder > curEventUpdateOrder) or (newEventUpdateOrder == curEventUpdateOrder and eventUpdate['update_key'] > curEventUpdate['update_key']):
                    updatesByEvent[eventUpdate['event_id']] = eventUpdate

        return updatesByEvent.values()

    def _formatEventPattern(self, pattern, params):
        return pattern.format(
            team1=params['team1'],
            team2=params['team2'],
            score=params['score'],
            time=params['time'],
            gender=params['gender'],
            Sport=params['Sport'],
            details=params['details'],
            Details=params['Details'],
            stage=params['stage'],
            Stage=params['Stage'],
        )

    def _formatMedalPattern(self, pattern, params):
        return pattern.format(
            Medal=params['Medal'],
            Name=params['Name'],
            Sport=params['Sport'],
            gender=params['gender'],
            Details=params['Details'],
        )

    def _getDateStr(self, timestamp):
        dt = datetime.datetime.fromtimestamp(timestamp)
        return '%d %s %d' % (dt.day, MONTH_NAMES_GENITIVE[self.lang][dt.month], dt.year)

    def _getTimeStr(self, timestamp):
        return datetime.datetime.fromtimestamp(timestamp).strftime('%H:%M') + ' (' + COMMON_STRINGS[self.lang]['msk_timezone'] + ')'

    def _getScoreStr(self, event):
        if (event['sport'] == 'hockey') and ('end_status' in event) and (event['end_status'] == 'shootout'):
            return '%s (%s)' % (event['score'], COMMON_STRINGS[self.lang]['by_shootout'])
        else:
            return event['score']

    def _getSerpRequest(self, eventUpdate):
        eventDate = self._getDateStr(eventUpdate['ts_start'])
        if eventUpdate['team1'] and eventUpdate['team2']:
            serpRequest = '%s %s %s %s %s' % (eventUpdate['sport_name'], eventUpdate['gender'], eventUpdate['team1'], eventUpdate['team2'], eventDate)
            if eventUpdate['update_key'] == 'in_progress':
                serpRequest += ' ' + COMMON_STRINGS[self.lang]['video']
        else:
            serpRequest = '%s %s %s %s %s' % (eventUpdate['sport_name'], eventUpdate['gender'], eventUpdate['details'], eventUpdate['stage'], eventDate)
        return u'олимпиада ' + serpRequest

    def _getLastGoalMinute(self, goals):
        if goals is None:
            return '0'
        if len(goals) == 0:
            return '0'
        lastGoal = sorted(goals, key=lambda goal: goal['id'])[-1]
        if lastGoal.get('time') is None:
            # Sometimes Sport-API returns empty time for a new goal.
            return 'XX'
        # "81'"    -> "81"
        # "90'+1'" -> "90_1"
        lastGoalMinute = re.sub('[^\d\+]', '', lastGoal['time'])
        return lastGoalMinute.replace('+', '_')

    def _getNewEventUpdatesKeysForEvent(self, prevEventState, curEventState):
        sportEvents = SPORT_EVENTS[curEventState['sport']]
        eventUpdates = []
        if curEventState['status'] and (curEventState['status'] in sportEvents) and (curEventState['status'] != prevEventState['status']):
            eventUpdates.append(curEventState['status'])
        if (curEventState['result'] and
            (curEventState['result'] != prevEventState['result']) and
            ((curEventState['result'][0] > prevEventState['result'][0]) or (curEventState['result'][1] > prevEventState['result'][1]))
        ):
            # Add goal minute into suffix, because goal may be cancelled. So the same score may be a different event
            eventUpdates.append('score_%d_%d_minute_%s' % (curEventState['result'][0], curEventState['result'][1], curEventState['last_goal_minute']))
        return eventUpdates

    def _getEventUpdatesForEvent(self, eventId, prevEventState, curEventState):
        curTs = time.time()
        newEventUpdates = [
            {'update_key': updateKey} for updateKey in self._getNewEventUpdatesKeysForEvent(prevEventState, curEventState)
        ]

        # Fill common fields
        for eventUpdate in newEventUpdates:
            if curEventState.get('result'):
                eventUpdate['score'] = '%d:%d' % (curEventState['result'][0], curEventState['result'][1])
            else:
                eventUpdate['score'] = '0:0'
            eventUpdate['team1'] = curEventState['team1']
            eventUpdate['team1_id'] = curEventState['team1_id']
            eventUpdate['team2'] = curEventState['team2']
            eventUpdate['team2_id'] = curEventState['team2_id']
            eventUpdate['ts_start'] = curEventState['ts_start']
            eventUpdate['event_id'] = eventId
            # internal sport var - hockey, curling, other
            eventUpdate['sport'] = curEventState['sport']
            eventUpdate['sport_code'] = curEventState['sport_code']
            eventUpdate['add_time'] = curTs
            eventUpdate['gender'] = COMMON_STRINGS[self.lang][curEventState['gender']]
            eventUpdate['sport_name'] = curEventState['sport_names'][self.lang]
            eventUpdate['details'] = curEventState['details']
            eventUpdate['stage'] = curEventState['stage']

        # Don't generate repeated events
        prevEventUpdatesKeys = map(
            lambda event: event['update_key'],
            prevEventState.get('event_updates', [])
        )
        newEventUpdates = filter(
            lambda event: event['update_key'] not in prevEventUpdatesKeys,
            newEventUpdates
        )

        return newEventUpdates

    def _updateMedals(self, catalog, tsNow):
        countries = ['OA']  # 'UA', 'BY', 'KZ']
        newMedals = []
        for country in countries:
            prevMedalIds = self.db.getMedalIds(country)
            newMedalIds = []
            medalsState = catalog.getCountryMedals(country)
            if not medalsState:
                continue

            for medal in medalsState:
                if not (medal["id"] in prevMedalIds or medal["event_id"] == "p942642"):  # skip first russian medal
                    personInfo = catalog.getPersonInfo(medal["person_id"])
                    eventInfo = catalog.getEventInfo(medal['event_id'])
                    if not eventInfo:
                        self._logMessage('Failed to get event info (for medal ' + str(medal["id"]) + ')', eventId=medal['event_id'])
                        continue
                    newMedalIds.append(medal["id"])

                    personName = u""
                    if personInfo:
                        personName = personInfo["name"][self.lang]
                    if (not personName) and personInfo:
                        personName = personInfo["name"]["ru"]
                    newMedals.append({
                            "id": medal["id"],
                            "event_id": medal["event_id"],
                            "medal_type": medal["medal"],
                            "person_id": medal["person_id"],
                            "person_name": personName,
                            "gender": COMMON_STRINGS[self.lang][eventInfo['gender']],
                            "sport_name": eventInfo['sport']['name'][self.lang],
                            "details": '. '.join([token for token in medal["competition"]["name"][self.lang].split(". ") if token != u"Женщины" and token != u"Мужчины"]),
                            "country": country,
                            "add_time": tsNow,
                            "medal_timestamp": medal["timestamp"]
                        })
            self.db.updateMedalIds(country, newMedalIds)
        self.db.addNewMedals(newMedals)

    def _updateEventsListAndMedals(self):
        catalog = OlympCatalog()
        tsNow = time.time()

        self._updateMedals(catalog, tsNow)

        for event in catalog.getEventsForPeriod(tsNow - self.checkEventsAge, tsNow + self.startAlertTime * 2):
            prevEventState = self.db.getEventState(event['id'])

            eventInfo = None
            try:
                eventInfo = catalog.getEventInfo(event['id'])
            except Exception as e:
                self._logMessage('Failed to get event info: ' + str(e), eventId=event['id'])

            if not eventInfo:
                # Sometimes SportAPI catalog contains strange matches, that are not found via getEventInfo
                continue

            if self.logOlympResponses and (event['ts_start'] < tsNow) and ((event.get('ts_finish') is None) or (event['ts_finish'] + HALF_HOUR > tsNow)):
                self.db.logOlympResponse(
                    respKey="%s/%s" % (get_sport(eventInfo['sport']['code']), event['id']),
                    respData=eventInfo
                )

            eventStatus = eventInfo['status']
            if (eventStatus == 'not_started') and (event['ts_start'] > tsNow):
                if (event['ts_start'] - tsNow <= self.startAlertTime):
                    eventStatus = 'need_start_alert'

            event_name_tokens = [token for token in event['name'][self.lang].split(". ") if token != u"Женщины" and token != u"Мужчины"]
            stage = ""
            details = event_name_tokens[0]
            if len(event_name_tokens) == 2:
                stage = event_name_tokens[1]
            if get_sport(eventInfo['sport']['code']) == "hockey":
                stage = event_name_tokens[0]
            curEventState = {
                'sport': get_sport(eventInfo['sport']['code']),
                'sport_code': eventInfo['sport']['code'],
                'sport_names': eventInfo['sport']['name'],
                'ts_start': event['ts_start'],
                'gender': event['gender'],
                'details': details,
                'status': eventStatus,
                'stage': stage,
                'last_goal_minute': self._getLastGoalMinute(eventInfo.get('goals'))
            }

            # for team sports
            if eventInfo.get('teams'):
                curEventState['team1'] = eventInfo['teams'][0]['name'][self.lang]
                curEventState['team1_id'] = eventInfo['teams'][0]['id']
                curEventState['team2'] = eventInfo['teams'][1]['name'][self.lang]
                curEventState['team2_id'] = eventInfo['teams'][1]['id']
                curEventState['result'] = [eventInfo['teams'][0]['result'], eventInfo['teams'][1]['result']]
                if (curEventState['result'][0] is None) or (curEventState['result'][1] is None):
                    curEventState['result'] = prevEventState['result']
            # for individual sports
            else:
                curEventState['team1'] = None
                curEventState['team1_id'] = None
                curEventState['team2'] = None
                curEventState['team2_id'] = None
                curEventState['result'] = None

            eventUpdates = self._getEventUpdatesForEvent(event['id'], prevEventState, curEventState)

            curEventState['event_updates'] = prevEventState.get('event_updates', [])
            curEventState['event_updates'].extend(map(
                lambda e: {'update_key': e['update_key'], 'add_time': e['add_time']},
                eventUpdates
            ))

            self.db.addNewEventUpdates(eventUpdates)
            self.db.setEventState(event['id'], curEventState)

        if self.logOlympResponses:
            self.db.clearOldOlympResponses()

    def _logMessage(self, msg, eventId=None):
        dateStr = datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')
        msg = dateStr + ' ' + msg.strip('\n') + '\n'
        if hasattr(self, 'logFile'):
            self.logFile.write(msg)
        else:
            print >>sys.stderr, msg
        # write into db log
        self.db.logSendPush(eventId=eventId, msg=msg)

    def _sendMedalPush(self, push):
        data = {
            "receiver": ["tag:sportEvent2=='olymp_medals.OA' && platform=='browser'"],
            "ttl": push["ttl"],
            "schedule": "now",
            "data": {
                "push_id": "olymp_medals.%s" % push["country"],
                "push_uri": "https://yandex.ru/search?text=%s&query_source=sport_push" % urllib.quote(u"олимпиада медальный зачет".encode("utf-8")),
                "push_action": "uri"
            },
            "notification": {
                "title": push["title"],
                "body": push["message"],
                "icon": "https://avatars.mds.yandex.net/get-sport/223098/ya_icon/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "spread_interval": 20,
            "project": "sup",
            "priority": "HIGH",
            "is_data_only": True,
            "transport": "Xiva"
        }
        headers = {
            "Content-type": "application/json;charset=UTF-8",
            "Authorization": "OAuth %s" % self.db.getSecurityVar("sup_oauth_key").encode("utf-8")
        }
        resp = requests.post(
            self.config['sendPushUrl'],
            data=json.dumps(data, ensure_ascii=False).encode('utf-8'),
            headers=headers,
            verify=False
        )
        self._logMessage('sendMedalPush, data: ' + json.dumps(data, ensure_ascii=False, indent=2).encode('utf-8'), eventId=push['event_id'])
        self._logMessage('sendMedalPush, status_code: ' + str(resp.status_code), eventId=push['event_id'])
        self._logMessage('sendMedalPush, serp_id: ' + str(json.loads(resp.text)["id"]), eventId=push['event_id'])
        resp.raise_for_status()  # raise in case of 4xx or 5xx

    def _sendMedalPushExp(self, push):
        data = {
            "receiver": ["yt://home/search-functionality/sup/ASSISTANT-1651/experiment"],
            "ttl": push["ttl"],
            "schedule": "now",
            "data": {
                "push_id": "olymp_medals.%s" % push["country"],
                "push_uri": "https://yandex.ru/search?text=%s&query_source=sport_push_ya" % urllib.quote(u"олимпиада медальный зачет".encode("utf-8")),
                "push_action": "uri"
            },
            "notification": {
                "title": push["title"],
                "body": push["message"],
                "icon": "https://avatars.mds.yandex.net/get-sport/223098/ya_icon/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "spread_interval": 20,
            "project": "sup",
            "priority": "HIGH",
            "is_data_only": True,
            "transport": "Xiva"
        }
        headers = {
            "Content-type": "application/json;charset=UTF-8",
            "Authorization": "OAuth %s" % self.db.getSecurityVar("sup_oauth_key").encode("utf-8")
        }
        resp = requests.post(
            self.config['sendPushUrl'],
            data=json.dumps(data, ensure_ascii=False).encode('utf-8'),
            headers=headers,
            verify=False
        )
        self._logMessage('sendMedalPushExp, data: ' + json.dumps(data, ensure_ascii=False, indent=2).encode('utf-8'), eventId=push['event_id'])
        self._logMessage('sendMedalPushExp, status_code: ' + str(resp.status_code), eventId=push['event_id'])
        self._logMessage('sendMedalPushExp, serp_id: ' + str(json.loads(resp.text)["id"]), eventId=push['event_id'])
        resp.raise_for_status()  # raise in case of 4xx or 5xx

    def _sendPushIntoSearchapp(self, push, pushLength):
        if pushLength != "long":
            return
        data = {
            "receiver": ["tag:(sportEvent2=='%s') && (app_id=='ru.yandex.mobile' || app_id=='ru.yandex.mobile.inhouse' || app_id LIKE 'ru.yandex.searchplugin%%')" % push["event_id"]],
            "ttl": push["ttl"],
            "schedule": "now",
            "data": {
                "push_id": "olymp_event." + push["sport_code"],
                "push_uri": "viewport://?sport_entities=" + urllib.quote("|") + "olymp_eid=%s&viewport_id=serp&query_source=sport_push&text=%s" % (push["event_id"], urllib.quote(push["serp_request"].encode("utf-8"))),
                "push_action": "uri"
            },
            "notification": {
                "title": push["title"],
                "body": push["message"],
                "icon": "http://avatars.mds.yandex.net/get-serp/23508/yandex_search_app_icon/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "android_features": {
                "ledType": 1,
                "soundType": 1
            },
            "spread_interval": 10,
            "project": "searchapp",
            "priority": "HIGH"
        }
        headers = {
            "Content-type": "application/json;charset=UTF-8",
            "Authorization": "OAuth %s" % self.db.getSecurityVar("sup_oauth_key").encode("utf-8")
        }
        resp = requests.post(
            self.config['sendPushUrl'],
            data=json.dumps(data, ensure_ascii=False).encode('utf-8'),
            headers=headers,
            verify=False
        )
        self._logMessage('sendPushIntoSearchapp, data: ' + json.dumps(data, ensure_ascii=False, indent=2).encode('utf-8'), eventId=push['event_id'])
        self._logMessage('sendPushIntoSearchapp, status_code: ' + str(resp.status_code), eventId=push['event_id'])
        self._logMessage('sendPushIntoSearchapp, sup_id: ' + str(json.loads(resp.text)["id"]), eventId=push['event_id'])
        resp.raise_for_status()  # raise in case of 4xx or 5xx

    def _sendPushIntoBrowser(self, push, pushLength):
        if pushLength != "short":
            return
        data = {
            "receiver": ["tag:sportEvent2=='%s' && platform=='browser'" % push["event_id"]],
            "ttl": push["ttl"],
            "schedule": "now",
            "data": {
                "push_id": "olymp_event." + push["sport_code"],
                "push_uri": "https://yandex.ru/search?text=%s&sport_entities=|olymp_eid=%s&query_source=sport_push" % (urllib.quote(push["serp_request"].encode("utf-8")), push["event_id"]),
                "push_action": "uri"
            },
            "notification": {
                "title": push["title"],
                "body": push["message"],
                "icon": "https://avatars.mds.yandex.net/get-sport/223098/ya_icon/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "spread_interval": 10,
            "project": "sup",
            "priority": "HIGH",
            "is_data_only": True,
            "transport": "Xiva"
        }
        headers = {
            "Content-type": "application/json;charset=UTF-8",
            "Authorization": "OAuth %s" % self.db.getSecurityVar("sup_oauth_key").encode("utf-8")
        }
        resp = requests.post(
            self.config['sendPushUrl'],
            data=json.dumps(data, ensure_ascii=False).encode('utf-8'),
            headers=headers,
            verify=False
        )
        self._logMessage('sendPushIntoBrowser, data: ' + json.dumps(data, ensure_ascii=False, indent=2).encode('utf-8'), eventId=push['event_id'])
        self._logMessage('sendPushIntoBrowser, status_code: ' + str(resp.status_code), eventId=push['event_id'])
        self._logMessage('sendPushIntoBrowser, serp_id: ' + str(json.loads(resp.text)["id"]), eventId=push['event_id'])
        resp.raise_for_status()  # raise in case of 4xx or 5xx

    def pushNewEvents(self):
        self.logFile = open(self.config['eventsPushLog'], 'a')

        eventsFromId = self.db.getClusterVar('last_eventUpdate_id') or 0
        medalFromId = self.db.getClusterVar('last_medal_id') or 0

        self._updateEventsListAndMedals()
        eventsUpdates, lastId = self.db.getLastEventUpdates(fromId=float(eventsFromId), fromTs=time.time() - self.eventsMaxAge)
        eventsUpdates = self._leaveLastUpdateForEachEvent(eventsUpdates)

        medals, lastMedalsId = self.db.getLastMedals(fromId=float(medalFromId), fromTs=time.time() - self.eventsMaxAge)

        medalPushes = []
        for medal in sorted(medals, key=lambda e: int(e['_id'])):
            medalParams = {
                "Medal": capitalize_first_letter(MEDAL_NAMES[medal["medal_type"]]),
                "Name": capitalize_first_letter(medal["person_name"]) + ". " if medal["person_name"] else u'',
                "Sport": capitalize_first_letter(medal["sport_name"]),
                "gender": u", " + medal["gender"] if medal["gender"] else u'',
                "Details": capitalize_first_letter(medal["details"])
            }

            medalPushes.append({
                '_id': medal['_id'],
                'event_id': medal['event_id'],
                'title': self._formatMedalPattern(MEDAL_MESSAGE[0], medalParams),
                'message': self._formatMedalPattern(MEDAL_MESSAGE[1], medalParams),
                'ttl': 4 * ONE_HOUR,
                'country': medal["country"],
            })
        for push in medalPushes:
            try:
                self._sendMedalPush(push)
            except Exception as e:
                # TODO: retry on connection errors? In this case ConnectionError exception should be raised
                self._logMessage('Failed to send medal push: ' + str(e), eventId=push['event_id'])
            try:
                self._sendMedalPushExp(push)
            except Exception as e:
                # TODO: retry on connection errors? In this case ConnectionError exception should be raised
                self._logMessage('Failed to send medal push exp: ' + str(e), eventId=push['event_id'])
            self.db.setClusterVar('last_medal_id', push['_id'])

        pushes = {"short": [], "long": []}
        for eventUpdate in sorted(eventsUpdates, key=lambda e: int(e['_id'])):
            updateType = self._updateKeyToType(eventUpdate['update_key'])
            updateTime = self._getTimeStr(eventUpdate['ts_start'])
            scoreStr = self._getScoreStr(eventUpdate)
            serpRequest = self._getSerpRequest(eventUpdate)
            messageKey = '%s_%s' % (eventUpdate['sport'], updateType)

            for pushLength in ["short", "long"]:
                if messageKey in EVENT_MESSAGES[self.lang][pushLength]:
                    pattern = EVENT_MESSAGES[self.lang][pushLength][messageKey]['default']
                    titlePattern, messagePattern = pattern[:2]
                    eventParams = {
                        'team1': eventUpdate['team1'],
                        'team2': eventUpdate['team2'],
                        'score': scoreStr,
                        'time': updateTime,
                        'gender': u', ' + eventUpdate['gender'] if eventUpdate['gender'] else u'',
                        'Sport': eventUpdate['sport_name'],
                        'details': uncapitalize_first_letter(eventUpdate['details']),
                        'Details': capitalize_first_letter(eventUpdate['details']),
                        'stage': u", " + uncapitalize_first_letter(eventUpdate['stage']) if eventUpdate['stage'] else u"",
                        'Stage': u". " + capitalize_first_letter(eventUpdate['stage']) if eventUpdate['stage'] else u"",
                    }
                    pushes[pushLength].append({
                        '_id': eventUpdate['_id'],
                        'event_id': eventUpdate['event_id'],
                        'sport_code': eventUpdate['sport_code'],
                        'title': self._formatEventPattern(titlePattern, eventParams),
                        'message': self._formatEventPattern(messagePattern, eventParams),
                        'serp_request': serpRequest.lower(),
                        'ttl': ONE_HOUR if updateType == 'finished' else 15 * ONE_MINUTE
                    })

        for pushLength in ["short", "long"]:
            for push in pushes[pushLength]:
                try:
                    self._sendPushIntoSearchapp(push, pushLength)
                except Exception as e:
                    # TODO: retry on connection errors? In this case ConnectionError exception should be raised
                    self._logMessage('Failed to send push: ' + str(e) + ' to searchapp', eventId=push['event_id'])
                try:
                    self._sendPushIntoBrowser(push, pushLength)
                except Exception as e:
                    # TODO: retry on connection errors? In this case ConnectionError exception should be raised
                    self._logMessage('Failed to send ' + pushLength + ' push: ' + str(e) + ' to browser', eventId=push['event_id'])

                self.db.setClusterVar('last_eventUpdate_id', push['_id'])

        self.logFile.close()
