# -*- coding: utf-8 -*-
from sandbox import sdk2
from sandbox import sandboxsdk
import logging

import kazoo
import sandbox.common.types.notification as ctn

import os
import time

from sandbox.projects.sport_wizard.pushes.lib.push_processor import PushProcessor
from sport_api_fetcher import get_WC2018_events
from sport_api_fetcher import get_match_info
from pushes_wc2018 import filter_event_info
from pushes_wc2018 import generate_pushes
from pushes_wc2018 import WC2018PushSender
from pushes_wc2018 import mongoConfig


logger = logging.getLogger(__name__)


class ZKClient(object):
    ZK_HOSTS = ('zookeeper1.search.yandex.net:2181,'
                'zookeeper2.search.yandex.net:2181,'
                'zookeeper3.search.yandex.net:2181,'
                'zookeeper4.search.yandex.net:2181,'
                'zookeeper5.search.yandex.net:2181')

    def __enter__(self):
        self.kz = kazoo.client.KazooClient(hosts=self.ZK_HOSTS, read_only=False)
        self.kz.start()
        return self.kz

    def __exit__(self, type, value, traceback):
        self.kz.stop()


class GenerateWC2018Pushes(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        oauth_key_name = sdk2.parameters.String('SUP oauth key vault name', required=True)
        yt_token_secret_name = sdk2.parameters.String('YT oauth key vault name', required=True)
        yt_server = sdk2.parameters.String('YT server short name', default="hahn")
        logs_table_path = sdk2.parameters.String('YT path for logs', default="//home/freshness/wc_2018_pushes_log/stream")

    class Requirements(sdk2.Task.Requirements):
        environments = (
            sandboxsdk.environments.PipEnvironment('yandex-yt'),
            sandboxsdk.environments.PipEnvironment('yandex-yt-yson-bindings-skynet'),
        )

    ZK_LOCK_PATH = '/freshness/wc2018_events_generate_lock'
    ZK_LOCK_TIMEOUT = 5

    def _execute(self):
        logFilename = "events.log"
        currentTs = int(time.time())
        with open(logFilename, "w") as f:
            oauth_key = sdk2.Vault.data(self.Parameters.oauth_key_name)
            mainTaskInstance = PushProcessor(get_WC2018_events, get_match_info, filter_event_info, generate_pushes, WC2018PushSender(oauth_key), mongoConfig["prod"], f.write, currentTs)
            mainTaskInstance.main()
        logLines = open(logFilename).read()
        if logLines:
            logging.info('Send email\n' + logLines)
            self.server.notification(
                subject="Olymp pushes sent",
                body=logLines,
                recipients=["iv-ivan@yandex-team.ru"],
                transport=ctn.Transport.EMAIL
            )
            try:
                from yt.wrapper import YtClient
                yt_token = sdk2.Vault.data(self.Parameters.yt_token_secret_name)
                client = YtClient(proxy=self.Parameters.yt_server, token=yt_token)
                if not client.exists(self.Parameters.logs_table_path):
                    client.create("map_node", self.Parameters.logs_table_path, recursive=True)
                client.write_table(os.path.join(self.Parameters.logs_table_path, str(currentTs)), [{"timestamp": currentTs, "value": logLines}])
            except:
                logging.info('No YT logs')
        else:
            logging.info('No data in events log')

    def on_execute(self):
        with ZKClient() as zk:
            lockPath = self.ZK_LOCK_PATH
            lock = zk.Lock(lockPath)
            try:
                if not lock.acquire(blocking=True, timeout=self.ZK_LOCK_TIMEOUT):
                    logging.info("Lock not aquired. Exit")
                    return
            except kazoo.exceptions.LockTimeout:
                logging.info("Lock not aquired. Exit")
                return

            try:
                return self._execute()
            finally:
                lock.release()
