#!/usr/bin/env python
# -*- coding: utf-8 -*-

import datetime
import requests
import urllib
import json
import copy
import urlparse

# ======= additional methods ===============

ONE_MINUTE = 60
ONE_HOUR = 60 * ONE_MINUTE
ONE_DAY = 24 * ONE_HOUR

COMMON_STRINGS = {
    "ru": {
        "msk_timezone": u"мск",
        "penalty": u"пен.",
        "ready_to_start_text": [u'{team1} — {team2}', u'Матч начнётся в {time} — не пропустите!'],
        "ready_to_start_text_with_translation": [u'{team1} — {team2}', u'Смотрите трансляцию Первого канала'],
        "start_text": [u'{team1} — {team2}', u'Матч начался'],
        "start_text_with_translation": [u'{team1} — {team2}', u'Матч начался — смотрите трансляцию'],
        "penalty_shootout_text": [u'{team1} — {team2}', u'Началась серия пенальти!'],
        "end_text": [u'{team1} — {team2}', u'Матч окончен со счётом {score}'],
        "goal_text": [u'Гол!', u'{team1} — {team2} {score}', u'{team1} — {team2}']
    }
}


def _getTimeStr(timestamp):
    return datetime.datetime.fromtimestamp(timestamp).strftime('%H:%M')


def _hasVideoTranslation(eventState):
    try:
        if "video" in eventState\
            and eventState["video"]:
                for item in eventState["video"]:
                    if item["type"] == "videoevents" and item["url"]:
                        return True, item["url"]
    except:
        pass
    return False, ""


def _isRussiaPlaying(eventState):
    try:
        if str(eventState["teams"][0]["id"]) == "78505" or str(eventState["teams"][1]["id"]) == "78505":
            return True
    except:
        pass
    return False


def get_score(eventState, lang):
    score = u"{}:{}".format(eventState["result"][0], eventState["result"][1])
    if eventState.get("result_additional") and sum(eventState["result_additional"]) > 0:
        score = score + u" ({}:{} {})".format(eventState["result_additional"][0], eventState["result_additional"][1], COMMON_STRINGS[lang]["penalty"])
    return score

# ========== main methods =============


mongoConfig = {
    'dev': {
        'mongoHost': 'localhost',  # behemoth.search.yandex.net
        'mongoPort': 27017,
        'databaseName': 'wc2018_events',
    },
    'prod': {
        'mongoReplicaSet': 'mongodb://man1-7436.search.yandex.net:22560,sas1-7558.search.yandex.net:29360,vla1-4211.search.yandex.net:22560/?replicaSet=sportpush',
        'mongoWriteReplicas': 2,
        'mongoWaitJournaling': True,
        'databaseName': 'wc2018_events',
    }
}


def filter_event_info(event):
    pass


class WC2018PushSender(object):
    def __init__(self, oauthKey):
        self.oauthKey = oauthKey
        self.searchapp_data = {
            "receiver": ["tag:((sportEvent2=='{eventId}') || (sportEvent LIKE '%%{eventId}%%')) && app_id IN ('ru.yandex.mobile', 'ru.yandex.mobile.inhouse', 'ru.yandex.searchplugin', 'ru.yandex.searchplugin.beta', 'ru.yandex.searchplugin.nightly', 'ru.yandex.searchplugin.dev')"],
            "ttl": ONE_HOUR,
            "schedule": "now",
            "data": {
                "push_id": "wc_2018_event",
                "push_uri": "viewport://?text={}&viewport_id=serp&query_source=sport_push",
                "push_action": "uri"
            },
            "notification": {
                "title": "{}",
                "body": "{}",
                "icon": "https://avatars.mds.yandex.net/get-bunker/135516/c2e400cc649d0853b6ea66813b474a66df51ec4c/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "android_features": {
                "ledType": 1,
                "soundType": 1
            },
            "project": "serp",
            "priority": "HIGH",
            "is_data_only": True,
        }

        self._browser_data = {
            "receiver": [],
            "ttl": ONE_HOUR,
            "schedule": "now",
            "data": {
                "push_id": "wc_2018_event",
                "push_uri": "https://yandex.ru/search?text={}&query_source=sport_push",
                "push_action": "uri"
            },
            "notification": {
                "title": "{}",
                "body": "{}",
                "icon": "https://avatars.mds.yandex.net/get-bunker/135516/c2e400cc649d0853b6ea66813b474a66df51ec4c/orig"
            },
            "throttle_policies": {
                "install_id": "sport_event_install_id",
                "device_id": "sport_event_device_id"
            },
            "project": "serp",
            "priority": "HIGH",
            "is_data_only": True,
            "transport": "Xiva"
        }

        self.yandex_browser_data = copy.deepcopy(self._browser_data)
        self.yandex_browser_data["receiver"].append("tag:((sportEvent2=='{eventId}') || (sportEvent LIKE '%%{eventId}%%')) && platform=='browser' && app_id=='YandexBrowser'")

        self.other_browser_data = copy.deepcopy(self._browser_data)
        self.other_browser_data["receiver"].append("tag:((sportEvent2=='{eventId}') || (sportEvent LIKE '%%{eventId}%%')) && platform=='browser' && app_id!='YandexBrowser'")

    def send_push(self, data, push, logFunction):
        headers = {
            "Content-type": "application/json;charset=UTF-8",
                "Authorization": "OAuth %s" % self.oauthKey.encode("utf-8")
        }
        resp = requests.post(
            "http://sup.yandex.net/pushes?priority=high",
            data=json.dumps(data, ensure_ascii=False).encode('utf-8'),
            headers=headers,
            verify=False
        )
        logFunction('SEND PUSH {} for event {}, data: '.format(push["push_id"], push["event_id"]) + json.dumps(data, ensure_ascii=False, indent=2).encode('utf-8'))
        logFunction('SEND PUSH {} for event {}, status_code: '.format(push["push_id"], push["event_id"]) + str(resp.status_code))
        logFunction('SEND PUSH {} for event {}, response: '.format(push["push_id"], push["event_id"]) + resp.text.encode('utf-8'))
        resp.raise_for_status()  # raise in case of 4xx or 5xx

    def __call__(self, push, timestamp, logFunction):
        exceptions = []
        for domain in ["ru", "not_ru"]:
            for i, dataClear in enumerate([self.searchapp_data, self.yandex_browser_data, self.other_browser_data]):
                data = copy.deepcopy(dataClear)
                # if push["has_translation"] and domain == "ru" and push["push_type"] == 4:
                #     continue #translation, don't send push about goals
                if push["has_translation"] and push["push_type"] in [0, 1] and domain == "ru" and push["isRussiaPlaying"]:
                    # morda send pushes in searchapp and YandexBrowser for anonses and beginning of matches of Russia team in russian region
                    if i != 2:
                        continue

                # EXPERIMENTS-21759
                if push["has_translation"] and push["push_type"] in [0, 1] and domain == "ru" and "exp_id" in push and i != 2:
                    continue

                if i == 0:
                    data["receiver"][0] = data["receiver"][0].format(eventId=push["event_id"]) + (" && geo_3==225" if domain == "ru" else " && geo_3!=225")
                else:
                    data["receiver"][0] = data["receiver"][0].format(eventId=push["event_id"]) + (" && web_tld==ru" if domain == "ru" else " && web_tld!=ru")
                if push["has_translation"] and domain == "ru" and push["push_type"] in [0, 1]:
                    data["notification"]["title"] = push["ru"]["text_with_translation"][0]
                    data["notification"]["body"] = push["ru"]["text_with_translation"][1]
                    data["data"]["push_uri"] = push["translation_url"] + "&query_source=sport_push"
                else:
                    data["notification"]["title"] = push["ru"]["text"][0]
                    data["notification"]["body"] = push["ru"]["text"][1]
                    data["data"]["push_uri"] = data["data"]["push_uri"].format(urllib.quote(push["ru"]["search_text"].encode("utf-8")), push["event_id"])

                if push["push_type"] == 4 and push["who"] and push["who"]["ru"]:
                    data["notification"]["title"] = data["notification"]["title"] + u" " + push["who"]["ru"]

                if i == 0 and push["push_type"] in [0, 1] and push["has_translation"] and domain == "ru" and not push["isRussiaPlaying"]:
                    # send push in app if russia is not playing, but firstly we have to fix translation uri and tags
                    videoParams = dict(urlparse.parse_qsl(urlparse.urlsplit(push["translation_url"]).query))
                    if "stream_id" not in videoParams:
                        continue
                    streamIdParam = videoParams["stream_id"]
                    initialReceiver = data["receiver"][0]

                    for tags, pushUriPrefix in zip([" && platform=='android' && app_version_code<6030000",
                                                    " && platform=='android' && app_version_code>=6030000 && app_version_code<6045000",
                                                    " && platform=='android' && app_version_code>=6045000",
                                                    " && platform=='ios' && app_version_code<4000000",
                                                    " && platform=='ios' && app_version_code>=4000000"],\
                                                   ["yellowskin://?primary_color=%23292a3a&secondary_color=%23ffffff&url=",
                                                    "yellowskin://?primary_color=%23fefefe&secondary_color=%23000000&url=",
                                                    "yellowskin://?buttons_color=%23ffffff&omnibox_color=%23191b24&status_bar_theme=dark&background_color=%23282a3a&text_color=%23ffffff?url=",
                                                    "yellowskin://?primary_color=%23292a3a&secondary_color=%23ffffff&url=",
                                                    "yellowskin://?buttons_color=%23ffffff&omnibox_color=%23191b24&status_bar_theme=dark&background_color=%23282a3a&text_color=%23ffffff?url="
                                                   ]):
                        data["receiver"][0] = initialReceiver + tags
                        translationUrl = pushUriPrefix + urllib.quote_plus("https://yandex.ru/portal/tvstream?from=appsearch&query_source=sport_push&stream_active=overlay&stream_id={}".format(streamIdParam))
                        data["data"]["push_uri"] = translationUrl
                        try:
                            self.send_push(data, push, logFunction)
                        except Exception, e:
                            exceptions.append(e)
                else:
                    try:
                        self.send_push(data, push, logFunction)
                    except Exception, e:
                        exceptions.append(e)
        if len(exceptions) > 0:
            for e in exceptions:
                logFunction('EXCEPTION while sendPush: {}'.format(str(e)))


def ReadyToStartPush(eventState, eventId):
    push = {"eventId": str(eventId), "priority": 0, "push_type": 0, "isRussiaPlaying": _isRussiaPlaying(eventState), "exp_id": "EXPERIMENTS-21759;83541;83542"}
    push["has_translation"], push["translation_url"] = _hasVideoTranslation(eventState)
    for lang in COMMON_STRINGS.keys():
        push[lang] = {}
        for k in ["text", "text_with_translation"]:
            push[lang][k] = copy.deepcopy(COMMON_STRINGS[lang]["ready_to_start_" + k])
            for i, text in enumerate(push[lang][k]):
                push[lang][k][i] = text.format(team1=eventState["teams"][0]["name"][lang],
                                   team2=eventState["teams"][1]["name"][lang],
                                   time=_getTimeStr(eventState["ts_start"]) + u' (' + COMMON_STRINGS[lang]['msk_timezone'] + u')')
        push[lang]["search_text"] = push[lang]["text"][0]
    return push


def StartPush(eventState, eventId):
    push = {"eventId": str(eventId), "priority": 1, "push_type": 1, "isRussiaPlaying": _isRussiaPlaying(eventState)}
    push["has_translation"], push["translation_url"] = _hasVideoTranslation(eventState)
    for lang in COMMON_STRINGS.keys():
        push[lang] = {}
        for k in ["text", "text_with_translation"]:
            push[lang][k] = copy.deepcopy(COMMON_STRINGS[lang]["start_" + k])
            for i, text in enumerate(push[lang][k]):
                push[lang][k][i] = text.format(team1=eventState["teams"][0]["name"][lang],
                                   team2=eventState["teams"][1]["name"][lang])
        push[lang]["search_text"] = push[lang]["text"][0]
    return push


def PenaltyShootoutPush(eventState, eventId):
    push = {"eventId": str(eventId), "priority": 3, "push_type": 2, "isRussiaPlaying": _isRussiaPlaying(eventState)}
    push["has_translation"], push["translation_url"] = _hasVideoTranslation(eventState)
    for lang in COMMON_STRINGS.keys():
        push[lang] = {}
        push[lang]["text"] = copy.deepcopy(COMMON_STRINGS[lang]["penalty_shootout_text"])
        for i, text in enumerate(push[lang]["text"]):
            push[lang]["text"][i] = text.format(team1=eventState["teams"][0]["name"][lang],
                               team2=eventState["teams"][1]["name"][lang])
        push[lang]["search_text"] = push[lang]["text"][0]
    return push


def EndPush(eventState, eventId):
    push = {"eventId": str(eventId), "priority": 4, "push_type": 3, "isRussiaPlaying": _isRussiaPlaying(eventState)}
    push["has_translation"], push["translation_url"] = _hasVideoTranslation(eventState)
    for lang in COMMON_STRINGS.keys():
        push[lang] = {}
        push[lang]["text"] = copy.deepcopy(COMMON_STRINGS[lang]["end_text"])
        for i, text in enumerate(push[lang]["text"]):
            push[lang]["text"][i] = text.format(team1=eventState["teams"][0]["name"][lang],
                               team2=eventState["teams"][1]["name"][lang],
                               score=get_score(eventState, lang))
        push[lang]["search_text"] = push[lang]["text"][0]
    return push


def GoalPush(eventState, eventId, goalsSum):
    push = {"eventId": str(eventId), "priority": 3, "push_type": 4, "isRussiaPlaying": _isRussiaPlaying(eventState)}
    push["has_translation"], push["translation_url"] = _hasVideoTranslation(eventState)
    push["who"] = None
    whoId = None
    try:
        if len(eventState["goals"]) >= goalsSum:
            push["who"] = eventState["goals"][goalsSum - 1]["name"]
            whoId = eventState["goals"][goalsSum - 1]["player_id"]
    except:
        pass  # no player name for goal
    try:
        if whoId and "lineup" in eventState:
            for player in eventState["lineup"]:
                if player["person"]["id"] == whoId:
                    push["who"] = player["person"]["name_short"]
    except:
        pass  # no short name for player

    for lang in COMMON_STRINGS.keys():
        push[lang] = {}
        push[lang]["text"] = copy.deepcopy(COMMON_STRINGS[lang]["goal_text"])
        for i, text in enumerate(push[lang]["text"]):
            push[lang]["text"][i] = text.format(team1=eventState["teams"][0]["name"][lang],
                               team2=eventState["teams"][1]["name"][lang],
                               score=get_score(eventState, lang))
        push[lang]["search_text"] = push[lang]["text"][2]
    return push


def generate_pushes(currentTimestamp, eventId, actualState, previousState):
    pushes = []
    if previousState is None or actualState is None:
        return []

    timeBeforeStart = int(actualState["ts_start"]) - int(currentTimestamp)

    if timeBeforeStart < 12 * ONE_MINUTE\
            and actualState["status"] == "not_started":
        actualState["status"] = "ready_to_start"

    if actualState.get("status") != previousState.get("status"):
        currentStatus = actualState.get("status")
        if currentStatus == "ready_to_start" and timeBeforeStart > 5 * ONE_MINUTE:
            pushes.append(ReadyToStartPush(actualState, eventId))
        elif currentStatus == "in_progress" and timeBeforeStart > -5 * ONE_MINUTE:
            pass  # NO PUSH HERE #pushes.append(StartPush(actualState, eventId))
        elif currentStatus == "penalty_shootout":
            pushes.append(PenaltyShootoutPush(actualState, eventId))
        elif currentStatus == "finished":
            if timeBeforeStart > -4 * ONE_HOUR\
                and (("ts_finish" in actualState\
                        and int(currentTimestamp) - int(actualState["ts_finish"]) < 10 * ONE_MINUTE)\
                    or "ts_finish" not in actualState):
                pushes.append(EndPush(actualState, eventId))

    if "result" in previousState and ("result" not in actualState or actualState["result"][0] is None or actualState["result"][1] is None):
        actualState["result"] = previousState["result"]
    try:
        if "result" in previousState and "result" in actualState\
            and actualState["result"] != previousState["result"]\
            and sum(actualState["result"]) > sum(previousState["result"]):
            pushes.append(GoalPush(actualState, eventId, sum(actualState["result"])))
    except:
        pass  # sometimes nulls in result (sport api) if match only begins

    return pushes
