import time
import datetime
import sys
import json
from pushes_database import PushesDatabase


def defaultLogFunction(msg):
    print >> sys.stderr, msg


class PushProcessor(object):
    def __init__(self, ActualEventsGetter, EventInfoGetter, EventInfoFilter, PushGenerator, PushSender, dbConfig, logFunction=None, timestamp=None):
        if timestamp:
            self.currentTime = int(timestamp)
        else:
            self.currentTime = int(time.time())
        if logFunction:
            self.logFunction = logFunction
        else:
            self.logFunction = defaultLogFunction
        self.db = PushesDatabase(self.currentTime, dbConfig)
        self.previousEvents = self.db.load_previous_actual_events()
        self.actualEventsUnmodified = {}
        self.pushes = []

        self.ActualEventsGetter = ActualEventsGetter
        self.EventInfoGetter = EventInfoGetter
        self.EventInfoFilter = EventInfoFilter
        self.PushGenerator = PushGenerator
        self.PushSender = PushSender

    def ts_to_strdate(self, ts):
        return datetime.datetime.fromtimestamp(int(ts)).strftime('%Y-%m-%d %H:%M:%S')

    def log_message(self, msg):
        if isinstance(msg, unicode):
            msg = msg.encode('utf-8')
        full_msg = self.ts_to_strdate(int(time.time())).encode('utf-8') + " " + msg + "\n"
        self.logFunction(full_msg)
        self.db.log_message(full_msg)

    def get_actual_events(self):
        # if exception - whole task will fail
        self.actualEvents = self.ActualEventsGetter(self.currentTime)

    def get_event_info(self, eventId):
        try:
            self.actualEvents[eventId].update(self.EventInfoGetter(eventId))
        except Exception, e:
            msg = "EXCEPTION: Can't LOAD EVENT_INFO for event {}, reason: {}".format(str(eventId), str(e))
            self.log_message(msg)

            if eventId in self.previousEvents:
                # try to get previous info about event
                self.actualEvents[eventId].update(self.previousEvents[eventId])
            else:
                # if no previous info - just skip event
                self.actualEvents.pop(eventId)

        if eventId in self.actualEvents:
            # filter unnecessary info
            try:
                self.EventInfoFilter(self.actualEvents[eventId])
            except Exception, e:
                msg = "EXCEPTION: Can't filter info for event {}, reason: {}".format(str(eventId), str(e))
                self.log_message()

    def generate_pushes(self, eventId, actualState, previousState):
        try:
            eventPushes = self.PushGenerator(self.currentTime, eventId, actualState, previousState)
        except Exception, e:
            msg = "EXCEPTION: Can't GENERATE PUSHES for event {}, reason: {}".format(str(eventId), str(e))
            self.log_message(msg)
            return

        if len(eventPushes) == 0:
            return

        priority_list = [push["priority"] for push in eventPushes if "priority" in push]
        if len(priority_list) == 0:
            max_priority = 0
        else:
            max_priority = max(priority_list)

        for push in eventPushes:
            push["event_id"] = eventId
            if not ("priority" in push) or push["priority"] == max_priority:
                push["to_send"] = True
            else:
                push["to_send"] = False
        self.pushes.extend(eventPushes)

    def send_pushes(self):
        for push in self.pushes:
            try:
                self.db.save_push(push)
                if push["to_send"]:
                    try:
                        self.PushSender(push, self.currentTime, self.log_message)
                        self.log_message("SEND PUSH: " + str(push["push_id"]))
                    except Exception, e:
                        msg = "EXCEPTION: Can't SEND PUSH {} for event {}, reason: {}".format(str(push["push_id"]), str(push["event_id"]), str(e))
                        self.log_message(msg)
                else:
                    self.log_message("NOT SEND PUSH: " + json.dumps(push))
            except Exception, e:
                msg = "EXCEPTION: Error while sending push {} for event {}, reason: {}".format(str(push["push_id"]), str(push["event_id"]), str(e))
                self.log_message(msg)

    def save_actual_events(self):
        self.db.save_current_actual_events(self.actualEvents)
        self.db.update_current_actual_events(self.actualEventsUnmodified)

    def main(self):
        self.get_actual_events()
        for eventId in self.actualEvents.keys():
            self.get_event_info(eventId)
        for eventId in self.actualEvents.keys():
            self.generate_pushes(eventId, self.actualEvents[eventId], self.previousEvents.get(eventId))
            if self.actualEvents[eventId] == self.previousEvents.get(eventId):
                self.actualEventsUnmodified[eventId] = self.actualEvents[eventId]
                self.actualEvents.pop(eventId)

        self.save_actual_events()
        self.db.inc_iteration()
        self.send_pushes()
