# coding: utf-8

import os
import shutil
import logging

from sandbox import sdk2
import sandbox.common.types.task as ctt
from sandbox.projects.common.build.YaMake2 import YaMake2
from sandbox.projects.sprav import resource_types as res
from sandbox.projects.common.ya_deploy import release_integration
import sandbox.projects.common.build.parameters as build_parameters


BUILD_TARGETS = [
    res.SPRAV_EDITOR_BINARY,
    res.ALTAY_EXPORT_PROXY_PACK,
    res.ALTAY_FEEDS_BINARY,
    res.SPRAV_PHOTO_API,
    res.SPRAV_SIMILAR_PERMALINKS_SEARCH,
    res.TYCOON_PHOTO_API,
    res.SPRAV_UNIFICATION_SERVER,
    res.SPRAV_SAAS_PUSH_CONFIGS,
    res.SPRAV_CALCULATE_SIMILARITY_SERVER,
    res.SPRAV_CALCULATE_SIMILARITY_SERVER_CONFIG,
    res.SPRAV_STREAM_SIGNALS_PUSH_LOADER,
]


class BuildSpravBinariesForDeploy(release_integration.ReleaseToYaDeployTask2, sdk2.Task):
    """
    Build sprav java packages for Ya.Deploy
    """

    class Parameters(sdk2.Parameters):
        svn_url = sdk2.parameters.ArcadiaUrl('Svn url for arcadia', required=True)

        # For arc builds
        use_aapi_fuse = build_parameters.UseArcadiaApiFuse()
        use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi()
        aapi_fallback = build_parameters.AllowArcadiaApiFallback()
        arc_secret = build_parameters.ArcSecret()

        with sdk2.parameters.CheckGroup("Resources to build") as build_targets:
            build_targets.choices = [(resource.name, resource.name) for resource in sorted(BUILD_TARGETS)]

    class Context(sdk2.Task.Context):
        build_tasks = dict()

    def on_execute(self):
        with self.memoize_stage.BUILD(commit_on_entrance=False):
            for res_name in self.Parameters.build_targets:
                resource = sdk2.Resource[res_name]
                task = YaMake2(
                    self,
                    description='Building {}'.format(resource),
                    checkout_arcadia_from_url=self.Parameters.svn_url,
                    use_aapi_fuse=self.Parameters.use_aapi_fuse,
                    use_arc_instead_of_aapi=self.Parameters.use_arc_instead_of_aapi,
                    aapi_fallback=self.Parameters.aapi_fallback,
                    arc_secret=self.Parameters.arc_secret,
                    build_system='ya',
                    build_type='release',
                    check_return_code=True,
                    strip_binaries=True,
                    sanitize=None,
                    targets=resource.arcadia_build_path,
                    arts=';'.join(resource.artifacts),
                    result_rt=res_name,
                    result_single_file=False,
                )
                task.save()
                task.enqueue()
                self.Context.build_tasks[res_name] = task.id
            raise sdk2.WaitTask(
                self.Context.build_tasks.values(),
                list(ctt.Status.Group.FINISH + ctt.Status.Group.BREAK),
                wait_all=True
            )
        with self.memoize_stage.COPY(commit_on_entrance=False):
            for resource_type, task_id in self.Context.build_tasks.items():
                resource = sdk2.Resource[resource_type]
                builded_resource = resource.find(task=sdk2.Task[task_id]).first()
                resource_path = str(sdk2.ResourceData(builded_resource).path)
                local_path = os.path.join('./', resource_type.lower())
                logging.info('Copying {} to {}'.format(resource_path, local_path))
                shutil.copytree(resource_path, local_path)
                current_resource = resource(
                    self,
                    resource.name,
                    local_path
                )
                current_resource.arcadia_branch = builded_resource.arcadia_branch
                current_resource.arcadia_revision = builded_resource.arcadia_revision
                current_resource.arcadia_tag = builded_resource.arcadia_tag
                current_resource.arcadia_trunk = builded_resource.arcadia_trunk
                resource_data = sdk2.ResourceData(current_resource)
                resource_data.ready()
