import json
import six
import time

from sandbox import sdk2
import sandbox.common.types.task as ctt
import sandbox.common.types.resource as ctr
from sandbox.projects.common.decorators import retries
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.nanny.client import NannyApiException


ALEMATE_RUNNING_GROUP = ('NEW', 'COMMITTED', 'MERGED')
ALEMATE_FAILED_GROUP = ('REJECTED', 'FAILED')
ALEMATE_SUCCESS_GROUP = ('DONE',)


class DeployNannyDashboardWithResources(sdk2.Task):
    """
        Deploy nanny dashboard recipe with given released resources
    """
    NANNY_API_URL = 'http://nanny.yandex-team.ru/'
    RETRIES = 10
    DELAY = 30

    class Requirements(sdk2.Task.Requirements):
        disk_space = 512  # Mb
        cores = 1

    class Parameters(sdk2.Task.Parameters):
        """
        Task parameters:
            :param component_resources: Dictionary of resources in the following format { resource_name: res_id, ...}
            :param yt_prefix: Altay db YT prefix
            :param yt_server: Name of YT cluster
            :param tag: Branch tag for the released resources
            :param vault_yt_token: YT token item name in secure vault
        """

        deployment_tasks = sdk2.parameters.List(label='List of sandbox task with release resources', required=True)
        nanny_dashboard_id = sdk2.parameters.String(label='Nanny dashboard id', required=True)
        nanny_dashboard_recipe_id = sdk2.parameters.String(label='Nanny dashboard recipe id', required=True)
        release_type = sdk2.parameters.String(label='Sandbox release type', required=True)
        nanny_token_vault_id = sdk2.parameters.String(label='Nanny token sandbox vault item id', required=True)

        poll_timeout = sdk2.parameters.Integer(label='Poll timeout for recipe deployment', default=120)

    def _get_release_tasks(self):
        return sorted([int(i.strip()) for i in self.Parameters.deployment_tasks])

    def _get_release_requests(self, nanny_client):
        res = {}
        for release_task in self._get_release_tasks():
            res[release_task] = self._get_release_request(nanny_client, release_task)

        return res

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_release_request(self, nanny_client, release_task):
        self.set_info('Find release request {}'.format(release_task))
        req = {
            'limit': 1,
            'query': {
                'sandboxTaskId': str(release_task),
                'sandboxReleaseType': [self.Parameters.release_type]
            },
        }
        res = nanny_client.find_releases(req)
        if len(res) > 0:
            return res[0]
        raise Exception("Couldn't find release request")

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_tickets(self, nanny_client, request_id, service):
        self.set_info('Get related ticket for {} release request and {} service'.format(request_id, service))
        find = {
            'limit': 1,
            'query': {
                'releaseId': request_id,
                'status': ['IN_QUEUE', 'COMMITTED', 'DEPLOY_SUCCESS'],
                'serviceId': service,
            }
        }
        res = nanny_client.filter_tickets(find)
        if len(res) > 0:
            return res[0]
        else:
            return None

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=NannyApiException)
    def _nanny_taskgroup_status(self, nanny_client, taskgroup):
        return nanny_client.get_taskgroup_status(taskgroup)['status']

    def _commit_ticket(self, nanny_client, ticket_id):
        self.set_info('Committing {} ticket'.format(ticket_id))
        payload = {
            'ticketId': ticket_id,
            'spec': {
                'type': 'COMMIT_RELEASE_REQUEST',
                'commitRelease': {}
            }
        }
        url = '{0}/api/tickets/CreateTicketEvent/'.format(nanny_client._api_url)
        return nanny_client._post_url(url, json.dumps(payload))

    def _run_dashboard_recipe(self, nanny_client, dashboard_name, recipe_name, snapshots):
        self.set_info('Run meta recipe {} for {} dashboard'.format(recipe_name, dashboard_name))
        return nanny_client.deploy_dashboard(
            dashboard_name,
            recipe_name,
            service_snapshots=snapshots
        )

    @retries(max_tries=RETRIES, delay=DELAY, exceptions=Exception)
    def _get_dashboard(self, nanny_client, dashboard):
        return nanny_client.get_dashboard(dashboard)

    def on_execute(self):
        nanny_client = nanny.NannyClient(
            api_url=self.NANNY_API_URL,
            oauth_token=sdk2.Vault.data(self.Parameters.nanny_token_vault_id)
        )

        recipe = nanny_client.get_dashboard_recipe(
            self.Parameters.nanny_dashboard_id,
            self.Parameters.nanny_dashboard_recipe_id
        )

        deployment_services = {}
        resources = {}

        for task in recipe['content']['tasks']:
            if task['data']['name'] == 'set_snapshot_target_state':
                service = task['data']['params']['service_id']
                deployment_services[service] = {}

                cur_resources = nanny_client.get_service_resources(service)
                for resource in cur_resources['content']['sandbox_files']:
                    if resource['resource_type'] in resources:
                        resources[resource['resource_type']].add(service)
                    else:
                        resources[resource['resource_type']] = {service}

        if not deployment_services:
            raise Exception('No services found for dashboard {} and recipe {}'.format(
                self.Parameters.nanny_dashboard_id,
                self.Parameters.nanny_dashboard_recipe_id
            ))

        releasing = []
        for task_id in self._get_release_tasks():
            released_task = self.server.task[task_id].read()

            if released_task['status'] != ctt.Status.RELEASED or released_task['release']['type'] != self.Parameters.release_type:
                if released_task['status'] not in ctt.Status.Group.FINISH + ctt.Status.Group.BREAK:
                    raise sdk2.WaitTask(
                        [released_task['id']],
                        [ctt.Status.Group.FINISH, ctt.Status.Group.BREAK],
                        wait_all=True
                    )

                if released_task['status'] not in {ctt.Status.RELEASED, ctt.Status.RELEASING}\
                        or released_task['release']['type'] != self.Parameters.release_type:
                    self.server.release(task_id=task_id, type=self.Parameters.release_type, subject='')
                releasing.append(task_id)
            else:
                for resource in sdk2.Resource.find(task=sdk2.Task[task_id], state=ctr.State.READY).limit(0):
                    if not resource.released:
                        continue
                    if resource.type not in resources:
                        raise Exception('Resource type {} not found in services'.format(resource.type))

                    for service in resources[resource.type]:
                        deployment_services[service][resource.type] = int(resource.id)

        if releasing:
            raise sdk2.WaitTask(releasing, [ctt.Status.RELEASED], wait_all=True)

        snapshots = {}
        for task, rq in six.iteritems(self._get_release_requests(nanny_client)):
            for service, resources in six.iteritems(deployment_services):
                cur_ticket = self._get_tickets(nanny_client, rq['id'], service)
                if not cur_ticket:
                    continue

                snapshot_id = cur_ticket['spec']['serviceDeployment']['snapshotId']
                if not snapshot_id:
                    continue

                snapshot_resources = nanny_client.get_history_runtime_attrs(snapshot_id)['content']['resources']
                service_resources = deployment_services[service]
                correct_snapshot = True
                for resource in snapshot_resources['sandbox_files']:
                    if resource['resource_type'] not in service_resources:
                        continue

                    if service_resources[resource['resource_type']] != int(resource['resource_id']):
                        correct_snapshot = False
                        break

                if correct_snapshot:
                    snapshots[service] = snapshot_id

        for service in deployment_services:
            if service not in snapshots:
                raise Exception('No snapshots with requested resources found for service {}'.format(service))

        deploy = self._run_dashboard_recipe(
            nanny_client,
            self.Parameters.nanny_dashboard_id,
            self.Parameters.nanny_dashboard_recipe_id,
            snapshots
        )

        deploy_taskgroup = deploy['taskgroup_id']

        while True:
            deploy_status = self._nanny_taskgroup_status(nanny_client, deploy_taskgroup)
            if deploy_status in ALEMATE_RUNNING_GROUP:
                self.set_info('Taskgroup {} is in state {}'.format(deploy_taskgroup, deploy_status))
                time.sleep(self.Parameters.poll_timeout)
            elif deploy_status in ALEMATE_FAILED_GROUP:
                raise Exception("Taskgroup {} is failed: {}".format(deploy_taskgroup, deploy_status))
            elif deploy_status not in ALEMATE_SUCCESS_GROUP:
                raise Exception("Taskgroup {} has unknown state: {}".format(deploy_taskgroup, deploy_status))
            else:
                break
