import json
import logging
import os
from sandbox import sdk2
from sandbox.projects.src_setup.RegularTests import SrcSetupRegularTestRequests
from sandbox.projects.websearch.upper import resources as upper_resources


class UpdateSrcSetupTestData(sdk2.Task):
    """Task that runs request_sampler to grep requests for new src_setup test data"""

    class Parameters(sdk2.Task.Parameters):
        request_sampler = sdk2.parameters.Resource(
            "request_sampler executable",
            resource_type=upper_resources.REQUEST_SAMPLER_EXECUTABLE,
            required=True,
        )

    def execute_with_logging(self, cmd):
        logging.debug('Running {}'.format(cmd))
        import subprocess
        s = subprocess.Popen(cmd, stderr=subprocess.PIPE, shell=True)
        exitcode = None
        while True:
            line = s.stderr.readline()
            exitcode = s.poll()

            if (not line) and (exitcode is not None):
                break
            logging.debug("pid %s: %s", os.getpid(), line.strip())

        logging.debug("pid %s: finished", os.getpid(), cmd)
        return exitcode

    def grep_requests(self, request_sampler_resource, graph, setup, test_name, requests_count):
        request_sampler_executable = str(sdk2.ResourceData(request_sampler_resource).path)
        requests_filename = '_'.join([graph, setup, test_name])
        cmd = [
            request_sampler_executable,
            '-s', setup,
            '--graph', graph,
            '-f', requests_filename,
            '-d', str(requests_count),
            '-t', sdk2.Vault.data("yt_token_for_eventlogs"),
            '--fresh'
        ]

        exit_code = self.execute_with_logging(' '.join(cmd))
        if exit_code != 0:
            logging.debug('request_samlper exited with {}'.format(str(exit_code)))
            raise Exception('request_sampler exited with non-zero return code!')

        return requests_filename

    def patch_ya_make(self, filename, resource_id):
        lines = open(filename, 'r').read().split('\n')
        result = []
        for line in lines:
            if line.startswith('    sbr://'):
                result.append('    sbr://{}'.format(resource_id))
            else:
                result.append(line)
        with open(filename, 'w') as f:
            f.write('\n'.join(result))

    def parse_config_file(self, config_filename):
        with open(config_filename, 'r') as f:
            json_config = json.load(f)
            return (json_config['enabled'], json_config['requests_count'])

    def on_execute(self):
        task_root_dir = os.path.abspath('.')

        logging.debug('Downloading Ya')
        self.execute_with_logging('svn cat svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/ya > ya')

        logging.debug('Chmoding Ya')
        self.execute_with_logging('chmod +x ya')

        logging.debug('Add Ya to PATH')
        self.execute_with_logging('export PATH=$PATH:{}'.format(task_root_dir))

        logging.debug('Checkouting Arcadia')
        self.execute_with_logging('./ya clone arcadia')

        logging.debug('Checkouting src_setup tests')
        self.execute_with_logging('./ya make --checkout -j0 arcadia/web/src_setup/tests/pseudo_src_setup')

        tests_local_path = os.path.abspath('arcadia/web/src_setup/tests/pseudo_src_setup')

        tests = []
        for vertical_name in os.listdir(tests_local_path):
            vertical_path = os.path.join(tests_local_path, vertical_name)
            if not os.path.isdir(vertical_path):
                continue

            for graph_name in os.listdir(vertical_path):
                graph_dir = os.path.join(vertical_path, graph_name)
                if not os.path.isdir(graph_dir):
                    continue

                for setup_name in os.listdir(graph_dir):
                    setup_dir = os.path.join(graph_dir, setup_name)
                    if not os.path.isdir(setup_dir):
                        continue

                    for test_name in os.listdir(setup_dir):
                        test_dir = os.path.join(setup_dir, test_name)
                        config_file = os.path.join(test_dir, 'updater_config.json')
                        if os.path.isfile(config_file):
                            tests.append((vertical_name, graph_name, setup_name, test_name))
                            logging.debug('Found test with vertical={}, graph = {}, setup={}, test_name = {}'.format(vertical_name, graph_name, setup_name, test_name))

        for test in tests:
            vertical, graph, setup, test_name = test
            full_test_name = '/'.join(test)
            test_dir = os.path.join(tests_local_path, full_test_name)

            logging.debug('Processing test {}'.format(full_test_name))

            is_enabled, requests_count = self.parse_config_file(os.path.join(test_dir, 'updater_config.json'))

            if is_enabled != 1:
                logging.debug('Test is disabled. Continue')
                continue

            if self.execute_with_logging('./ya make -rA --checkout --yt-store {}'.format(test_dir)) != 0:
                logging.debug('Test {} is broken. Skiping.'.format(full_test_name))
                continue

            logging.debug('Running request_sampler')

            try:
                requests_file = self.grep_requests(self.Parameters.request_sampler, graph, setup, test_name, requests_count)
            except:
                logging.debug('Failed to sample requests!')
                continue

            logging.debug('Uploading requests')
            requests_resource = SrcSetupRegularTestRequests(self, 'Requests for {} src_setup test'.format(test_name), requests_file)
            resource_data = sdk2.ResourceData(requests_resource)
            logging.debug('Make resource ready')
            resource_data.ready()

            logging.debug('Requests uploaded, resource_id={}'.format(requests_resource.id))

            test_dir = os.path.join(tests_local_path, '/'.join(test))
            ya_make = os.path.join(test_dir, 'ya.make')
            logging.debug('Patching ya.make')
            self.patch_ya_make(ya_make, requests_resource.id)

            logging.debug('Canonizing test')
            if self.execute_with_logging('./ya make -rAZ {}'.format(test_dir)) != 0:
                logging.debug('Failed to canonize test. Reverting')
                self.execute_with_logging('./ya svn revert --recursive {}'.format(test_dir))
                continue

            logging.debug('Successfuly canonized {}'.format(test_name))

        logging.debug('Finished updating tests. Commiting')
        if self.execute_with_logging('./ya svn commit arcadia -m \'Updating tests. Task: https://sandbox.yandex-team.ru/task/{}/view\''.format(self.id)) != 0:
            raise Exception('Failed to commit!')

        logging.debug('Done.')
