# coding: utf-8

from sandbox import sdk2

import sandbox.common.types.task as ctt
import requests
import json
import time
from requests.adapters import HTTPAdapter
from requests.packages.urllib3.util.retry import Retry
from sandbox.projects.common.nanny import nanny


class UidsFileResource(sdk2.Resource):
    """ Uids file resource"""
    releasable = True


class PrepareUidsFile(nanny.ReleaseToNannyTask2, sdk2.Task):
    """
        Prepare uids file for staff login checker balancer module
    """

    BLACKBOX_LINK = 'http://blackbox-mimino.yandex.net/blackbox?method=userinfo&format=json&userip=12.12.12.12&aliases=13&login={login}'

    POST_PARAMS = {
        '_fields': 'yandex.login,uid',
        'official.is_robot': 'False',
        'official.is_dismissed': 'False',
        '_limit': 1000000,
    }

    STAFF_ALIAS = '13'

    BB_CLIENT_ID = 239
    SELF_CLIENT_ID = 2011974

    class Parameters(sdk2.Parameters):
        release_on_success = sdk2.parameters.Bool('Release if the task complete success', default=False, required=True)
        with release_on_success.value[True]:
            release_status = sdk2.parameters.Bool('Release STABLE if true else TESTING', default=False, required=True)

    def get_service_ticket(self):
        from ticket_parser2.api.v1 import ServiceContext
        # Getting TVM keys
        tvm_keys = requests.get('https://tvm-api.yandex.net/2/keys?lib_version=2.3.0').content
        secret = sdk2.Vault.data("tvm_secret_for_prepare_uids_file_job")

        # Create context
        service_context = ServiceContext(self.SELF_CLIENT_ID, secret, tvm_keys)

        # Getting ticket
        ts = int(time.time())
        service_ticket = requests.post(
            'https://tvm-api.yandex.net/2/ticket/',
            data={
                'grant_type': 'client_credentials',
                'src': self.SELF_CLIENT_ID,
                'dst': self.BB_CLIENT_ID,
                'ts': ts,
                'sign': service_context.sign(ts, self.BB_CLIENT_ID)
            }
        ).json()
        return service_ticket[str(self.BB_CLIENT_ID)]['ticket']

    def on_execute(self):

        def requests_retry_session(retries=3, backoff_factor=0.3, status_forcelist=(500, 502, 504), session=None):
            session = session or requests.Session()
            retry = Retry(
                total=retries,
                read=retries,
                connect=retries,
                backoff_factor=backoff_factor,
                status_forcelist=status_forcelist,
            )
            adapter = HTTPAdapter(max_retries=retry)
            session.mount('http://', adapter)
            session.mount('https://', adapter)
            return session

        service_ticket = self.get_service_ticket()
        r = requests_retry_session().post(
            'http://staff-api.yandex-team.ru/v3/persons',
            headers={
                'Authorization': 'OAuth %s' % sdk2.Vault.data('robot_oauth_token'),
            },
            data=self.POST_PARAMS,
        )

        json_resp = r.json()
        staff = json_resp['result']

        uids_list = []
        for person in staff:
            login = person['yandex']['login']

            if login is None:
                continue

            login = requests.utils.quote(login, safe='@')
            full_url = self.BLACKBOX_LINK.format(login=login)
            # Attention! Make sure request running in one thread
            r = requests_retry_session().get(full_url, headers={'X-Ya-Service-Ticket': service_ticket}, timeout=3)

            resp = r.json()

            user = resp['users'][0]
            uid = user['id']

            if uid is None or uid == '':
                continue

            alias = user['aliases'].get(self.STAFF_ALIAS, '')

            if alias == '':
                continue

            uids_list.append(uid)

        with open('uids.txt', mode='w') as f:
            json.dump(uids_list, f)

        sdk2.ResourceData(UidsFileResource(
            self,
            'Result file',
            'uids.txt',
        ))

    def on_success(self, prev_status):
        sdk2.Task.on_success(self, prev_status)
        if self.Parameters.release_on_success:
            status = ctt.ReleaseStatus.STABLE if self.Parameters.release_status else ctt.ReleaseStatus.TESTING
            self.on_release(dict(
                releaser=self.author,
                release_status=status,
                release_subject='Preparing file with users uids',
                email_notifications=dict(to=['valgushev'], cc=[]),
                release_comments='preparing uids file',
            ))

    def on_release(self, params):
        super(PrepareUidsFile, self).on_release(params)
        self.mark_released_resources(params["release_status"])
