# -*- coding: utf-8 -*-

from __future__ import absolute_import, print_function, division

import os

import json
import threading
import signal
import logging

from sandbox import sdk2

logger = logging.getLogger(__name__)


class ProcessSupervisor(object):
    def __init__(self, command, env=None):
        self._command = command
        self._env = env

        self._running = False
        self._process = None  # type: sdk2.helpers.subprocess.Popen
        self._logger = None  # type: sdk2.helpers.process.ProcessLog
        self._bridge_thread = None  # type: threading.Thread

        self._progress = 0
        self._stderr = ''

    def run(self):
        logger.info('starting %r', self._command)

        assert not self._running

        self._running = True

        self._logger = sdk2.helpers.process.ProcessLog(logger=logger)

        self._process = sdk2.helpers.subprocess.Popen(
            self._command,
            stdout=self._logger.stdout,
            stderr=sdk2.helpers.subprocess.PIPE,
            shell=True,
            preexec_fn=os.setsid,
            env=self._env,
        )

        logger.info('starting bridge thread')

        self._bridge_thread = threading.Thread(target=self._read_stderr)
        self._bridge_thread.setDaemon(True)
        self._bridge_thread.start()

    def get_progress(self):
        return self._progress

    def get_return_code(self):
        if self._running or self._process is None:
            return None
        else:
            return self._process.returncode

    def is_running(self):
        return self._running

    def get_output(self):
        return self._stderr

    def terminate_process(self):
        logger.info('terminating subprocess')

        if not self._running:
            logger.info('subprocess is not running')
            return

        logger.info('sending SIGINT')
        os.killpg(os.getpgid(self._process.pid), signal.SIGINT)
        try:
            self._process.wait(timeout=120)
        except sdk2.helpers.subprocess.TimeoutExpired:
            logger.info('SIGINT timed out, sending SIGKILL')
            os.killpg(os.getpgid(self._process.pid), signal.SIGKILL)

        logger.info('waiting for subprocess')
        self._process.wait()
        logger.info('subprocess finished, waiting for bridge thread')
        self._bridge_thread.join()
        logger.info('bridge thread finished')

        self._running = False

    def _read_stderr(self):
        try:
            logger.info('bridge thread started')
            while True:
                line = self._process.stderr.readline()
                if not line:
                    break
                try:
                    data = json.loads(line)
                except ValueError:
                    self._stderr += line
                    logger.error('Cannot decode log line %r', line)
                else:
                    logger.info(data)
                    if 'progress' in data and 0 <= data['progress'] <= 1:
                        self._progress = data['progress']
                    if 'message' in data:
                        self._stderr += data['message'] + '\n'
            self._process.wait()
            self._running = False
        except Exception:
            logger.exception('error when handling process output')
            self.terminate_process()
            raise
